#!/usr/bin/env python
# -*- coding: utf-8 -*-

# wxZ3D-OSM.py
#
# Die von einer Z-EDGE Z3D Dashcam erzeugten 5-Minuten-Videos "im Stück"
# abspielen und parallel dazu die jeweiligen Geo-Koordinaten auf einer
# OSM-Karte mitzuführen.
#
# Detlev Ahlgrimm   02.2022
#
# 1.1.3     12.03.2022    D.A.    Slider nachgerüstet

import os
import sys
import time
import datetime
import wx
import threading
import subprocess
import Queue

from map_parse import FilesInFolder, ParseMap, SingleFileData
from osm_disp import MapWindow, MapFrame
from OSMhelper import OSMhelper

VERSION="1.1.3"

# ----------------------------------------------------------------------
# 
class Z3dOsmPanel(wx.Panel):
    def __init__(self, parent):
        wx.Panel.__init__(self, parent, wx.ID_ANY, style=wx.WANTS_CHARS)
        self.parent=parent
        self.osmWin=None
        self.mplayer_process=None
        self.cur_video_data=None
        self.tour_dist_km=None
        self.slider_grabbed=False

        sizer=wx.GridBagSizer(4, 4)
        sb_file_info_sizer=self.buildFileInfoArea()
        self.b_sel_folder=wx.Button(self, wx.ID_ANY, "select tour folder")
        self.b_sel_folder.Bind(wx.EVT_BUTTON, self.sel_folder)
        sb_tour_sizer=self.buildTourInfoArea()

        fix_sizer=wx.BoxSizer(wx.HORIZONTAL)
        self.delta_ctrl=wx.SpinCtrl(self, value="0", min=-5, max=10)
        fixtxt=wx.StaticText(self, wx.ID_ANY, " Fix [Sec.]")
        fix_sizer.Add(self.delta_ctrl)
        fix_sizer.Add(fixtxt)
        sb_video_sizer=self.buildVideoControlArea()
        sizer.Add(sb_file_info_sizer, (0, 0), (1, 1), wx.LEFT|wx.TOP|wx.RIGHT, 4)
        sizer.Add(self.b_sel_folder,  (0, 1), (1, 1), wx.LEFT|wx.TOP|wx.RIGHT|wx.ALIGN_CENTER_VERTICAL, 4)
        sizer.Add(sb_tour_sizer,      (1, 0), (1, 1), wx.LEFT|wx.TOP|wx.RIGHT, 4)
        sizer.Add(fix_sizer,          (1, 1), (1, 1), wx.LEFT|wx.TOP|wx.RIGHT|wx.ALIGN_CENTER_VERTICAL, 4)
        sizer.Add(sb_video_sizer,     (2, 0), (1, 3), wx.LEFT|wx.TOP|wx.RIGHT, 4)
        self.SetSizer(sizer)
        sizer.Fit(self)
        self.parent.Bind(wx.EVT_CLOSE, self.onClose)

        self.folder=None
        self.playlist_name="/tmp/playlist.txt"
        self.mplayer_prev_sec=None                                      # nur ein Merker, um nur einmal pro Sekunde aktualisieren zu müssen
        self.cur_file_index_in_playlist=None

        self.cmd_lock=threading.Lock()

        self.hlp=OSMhelper()
        self.timer=wx.Timer(self)
        self.Bind(wx.EVT_TIMER, self.OnTimer)

        self.mplayer_pause=False                                        # zum togglen des Pause-Modus
        self.mplayer_cur_sec=None                                       # die derzeitige Sekunde im aktuellen Video

    # ------------------------------------------------------------------
    # Legt die folgenden Controls an:
    #   self.cur_file_name_ctrl
    #   self.cur_file_data_ctrl
    #   self.cur_video_data_ctrl
    def buildFileInfoArea(self):
        sb_file=wx.StaticBox(self, wx.ID_ANY, " Video File ", size=(530, -1))
        sb_file_info_sizer=wx.StaticBoxSizer(sb_file, wx.HORIZONTAL|wx.EXPAND)
        self.cur_file_name_ctrl=wx.StaticText(self, wx.ID_ANY, "no file      ( ? / ? )")
        sb_file_info_sizer.Add(self.cur_file_name_ctrl,  0, wx.ALL, 5)

        self.cur_file_data_ctrl=wx.StaticText(self, wx.ID_ANY, "Length: %3d sec.      Current: %3d sec.      Percent: %3d %%"%(0, 0, 0))
        sb_file_info_sizer.Add(self.cur_file_data_ctrl,  0, wx.ALL, 5)

        self.cur_video_data_ctrl=wx.StaticText(self, wx.ID_ANY, "GPS-Location: %8.5f %7.5f      Speed: %3d km/h      Timestamp: %s"%(0.0, 0.0, 0, "-"))
        sb_file_info_sizer.Add(self.cur_video_data_ctrl,  0, wx.ALL, 5)
        return sb_file_info_sizer

    # ------------------------------------------------------------------
    # Legt die folgenden Controls an:
    #   self.tour_total_ctrl
    #   self.tour_cur_pos_ctrl
    def buildTourInfoArea(self):
        sb_tour=wx.StaticBox(self, wx.ID_ANY, " Tour Data ", size=(530, -1))
        sb_tour_info_sizer=wx.StaticBoxSizer(sb_tour, wx.HORIZONTAL|wx.EXPAND)

        self.tour_total_ctrl=wx.StaticText(self, wx.ID_ANY, "Start: --:--:-- --.--.----      End: --:--:-- --.--.----     Total: ---.- km / --:--:-- h")
        sb_tour_info_sizer.Add(self.tour_total_ctrl,  0, wx.ALL, 5)

        self.tour_cur_pos_ctrl=wx.StaticText(self, wx.ID_ANY, "Current:   0.00 km / --:--:-- h      Percent:   0% /   0 %")
        sb_tour_info_sizer.Add(self.tour_cur_pos_ctrl,  0, wx.ALL, 5)
        return sb_tour_info_sizer

    # ------------------------------------------------------------------
    # Legt die Button zur Video-Steuerung an.
    def buildVideoControlArea(self):
        sb_video=wx.StaticBox(self, wx.ID_ANY, " Video Control ")
        h_video_sizer=wx.StaticBoxSizer(sb_video, wx.VERTICAL)
        v_video_sizer=wx.BoxSizer(wx.HORIZONTAL)
        self.b_prev_file=wx.Button(self, wx.ID_ANY, "< file")
        self.b_minus_30= wx.Button(self, wx.ID_ANY, "<< 30")
        self.b_minus_5=  wx.Button(self, wx.ID_ANY, "<< 5")
        self.b_pause=    wx.Button(self, wx.ID_ANY, "pause")
        self.b_plus_5=   wx.Button(self, wx.ID_ANY, ">> 5")
        self.b_plus_30=  wx.Button(self, wx.ID_ANY, ">> 30")
        self.b_next_file=wx.Button(self, wx.ID_ANY, "> file")

        self.b_prev_file.Disable()
        self.b_minus_30.Disable()
        self.b_minus_5.Disable()
        self.b_pause.Disable()
        self.b_plus_5.Disable()
        self.b_plus_30.Disable()
        self.b_next_file.Disable()

        self.b_prev_file.Bind(wx.EVT_BUTTON, self.prev_file)
        self.b_minus_30.Bind( wx.EVT_BUTTON, self.minus_30)
        self.b_minus_5.Bind(  wx.EVT_BUTTON, self.minus_5)
        self.b_pause.Bind(    wx.EVT_BUTTON, self.pause)
        self.b_plus_5.Bind(   wx.EVT_BUTTON, self.plus_5)
        self.b_plus_30.Bind(  wx.EVT_BUTTON, self.plus_30)
        self.b_next_file.Bind(wx.EVT_BUTTON, self.next_file)

        v_video_sizer.Add(self.b_prev_file,  0, wx.ALL, 5)
        v_video_sizer.Add(self.b_minus_30,   0, wx.ALL, 5)
        v_video_sizer.Add(self.b_minus_5,    0, wx.ALL, 5)
        v_video_sizer.Add(self.b_pause,      0, wx.ALL, 5)
        v_video_sizer.Add(self.b_plus_5,     0, wx.ALL, 5)
        v_video_sizer.Add(self.b_plus_30,    0, wx.ALL, 5)
        v_video_sizer.Add(self.b_next_file,  0, wx.ALL, 5)
        h_video_sizer.Add(v_video_sizer,  0, wx.ALL, 2)

        self.slider=wx.Slider(self, value=0, minValue=0, maxValue=100, size=(670, -1), style=wx.SL_LABELS)
        self.slider.Disable()
        self.slider.Bind(wx.EVT_SLIDER, self.sliderGrabbed)
        self.slider.Bind(wx.EVT_SCROLL_CHANGED, self.sliderUpdateDone)
        h_video_sizer.Add(self.slider,  0, wx.ALL, 0)
        return h_video_sizer

    # ------------------------------------------------------------------
    # Aktualisiert die folgenden Controls:
    #   self.cur_file_name_ctrl
    #   self.cur_file_data_ctrl
    #   self.cur_video_data_ctrl
    def updateFileData(self):
        if self.mplayer_filename is not None:
            self.cur_file_name_ctrl.SetLabel("%s      ( %d / %d )"%(self.mplayer_filename, self.cur_file_index_in_playlist[0], self.cur_file_index_in_playlist[1]))

        if self.mplayer_file_length is not None and self.mplayer_cur_sec is not None:
            if self.mplayer_file_length==0:
                print("no mplayer_file_length")
                return
            cur_file_data=(self.mplayer_file_length, self.mplayer_cur_sec, 100/self.mplayer_file_length*self.mplayer_cur_sec)
            self.cur_file_data_ctrl.SetLabel("Length: %3d sec.      Current: %3d sec.      Percent: %3d %%"%cur_file_data)

        if self.cur_video_data is not None:
            self.cur_video_data_ctrl.SetLabel("GPS-Location: %8.5f %7.5f      Speed: %3d km/h      Timestamp: %s"% \
                            (self.cur_video_data["lat"], self.cur_video_data["lon"], float(self.cur_video_data["spd"]), self.cur_video_data["dt"]))
        else:
            self.cur_video_data_ctrl.SetLabel("GPS-Location: %8.5f %7.5f      Speed: %3d km/h      Timestamp: %s"%(0.0, 0.0, 0, "?"))

    # ------------------------------------------------------------------
    # Aktualisiert die folgenden Controls:
    #   self.tour_total_ctrl
    #   self.tour_cur_pos_ctrl
    def updateTourData(self):
        t0, te, d=self.tour_times
        t0s=datetime.datetime.strftime(t0, "%H:%M:%S %d.%m.%Y")
        tes=datetime.datetime.strftime(te, "%H:%M:%S %d.%m.%Y")
        tot=datetime.timedelta(seconds=d)
        self.tour_total_ctrl.SetLabel("Start: %s      End: %s     Total: %s km / %s h"%(t0s, tes, round(self.tour_dist_km, 2), tot))

        if self.cur_video_data is not None:
            tc=datetime.datetime.strptime(self.cur_video_data["dt"], "%H:%M:%S %d.%m.%Y")
            dist=self.track_dict[(self.cur_video_data["lat"], self.cur_video_data["lon"])]
            secs=(tc-t0).total_seconds()
            dur=datetime.timedelta(seconds=secs)
            try:
                pt=100/d*secs
            except:
                pt=0
            try:
                pd=100/self.tour_dist_km*dist
            except:
                pd=0
            self.tour_cur_pos_ctrl.SetLabel("Current: %5.2f km / %s h      Percent: %3d %% / %3d %%"%(dist, dur, pd, pt))
            self.slider.SetValue(secs)

    # ------------------------------------------------------------------
    # Formatiert die "map_data" so um, dass wxOSM sie als Track
    # verarbeiten und anzeigen kann.
    # Im Data-Bereich des Tracks wird die bisher zurückgelegte Strecke
    # in Kilometer eingetragen.
    def buildTrackFromMapData(self):
        trk=list()
        trk_dict=dict()
        p0=None
        dist=0.0
        for fn in sorted(self.map_data):
            for ln_nr in sorted(self.map_data[fn].line_dict):
                lat=self.map_data[fn].line_dict[ln_nr]["lat"]
                lon=self.map_data[fn].line_dict[ln_nr]["lon"]
                if p0:
                    dist+=self.hlp.haversine(p0, (lat, lon))
                p0=(lat, lon)
                trk.append(((lat, lon), dist))
                trk_dict.update({(lat, lon): dist})
        return trk, trk_dict

    # ------------------------------------------------------------------
    # Lädt die map-Dateien, erstellt die Playlist und schaltet die
    # Steuer-Buttons frei.
    def initAfterFolderSelection(self):
        if self.folder:
            fif=FilesInFolder()
            fif.buildPlaylist(self.folder, self.playlist_name)
            _, self.mp4_files=fif.getNames(self.folder)                 # alle mp4-Dateinamen sortiert als Liste
            self.mapParser=ParseMap(self.folder)
            self.map_data=self.mapParser.parseAll()                     # Format: {   mp4-file: SingleFileData, mp4-file: SingleFileData, ...}
            self.tour_times=self.mapParser.getTourTimes()

            self.b_prev_file.Enable()
            self.b_minus_30.Enable()
            self.b_minus_5.Enable()
            self.b_pause.Enable()
            self.b_plus_5.Enable()
            self.b_plus_30.Enable()
            self.b_next_file.Enable()
            self.b_sel_folder.Disable()

            self.slider.SetMax(self.tour_times[2])
            self.slider.Enable()

    # ------------------------------------------------------------------
    # Einmalig beim Greifen des Sliders den Pausemodus aktivieren.
    # Während des Slidens die Map nachführen.
    def sliderGrabbed(self, event):
        if not self.slider_grabbed:                                     # nur wenn der Slider frisch gegriffen wurde...
            self.pause_on()                                             # Video pausieren
        self.slider_grabbed=True

        wanted_sec=self.slider.GetValue()
        playlist_pos=self.mapParser.getFileForSec(wanted_sec)
        if playlist_pos is None:
            return
        mp4_fn, map_fn, start_sec, end_sec=playlist_pos
        if wanted_sec-start_sec in self.map_data[mp4_fn].line_dict:     # Zeit in map-Daten suchen
            cur_video_data=self.map_data[mp4_fn].line_dict[wanted_sec-start_sec]
            self.osmWin.centerPos(cur_video_data["lat"], cur_video_data["lon"])

    # ------------------------------------------------------------------
    # Wird einmalig nach manueller Änderung der Slider-Position
    # aufgerufen und bewegt die Position im Video zur gewünschten
    # Stelle.
    def sliderUpdateDone(self, event):
        wanted_sec=self.slider.GetValue()
        playlist_pos=self.mapParser.getFileForSec(wanted_sec)
        if playlist_pos is None:
            return
        self.timer.Stop()
        mp4_fn, map_fn, start_sec, end_sec=playlist_pos
        print("goto", wanted_sec, mp4_fn, start_sec, end_sec)
        idx_s, idx_e=self.mp4_files.index(self.mplayer_filename), self.mp4_files.index(mp4_fn)
        if idx_s<idx_e:
            for i in range(idx_s, idx_e):
                self.next_file()
        elif idx_s>idx_e:
            for i in range(idx_e, idx_s):
                self.prev_file()
        else:
            pass    # keine andere Datei
        if start_sec<wanted_sec:
            secs2skip=wanted_sec-start_sec
            print("secs2skip", secs2skip)
            self.skip_secs(secs2skip)
        self.pause_off()
        self.timer.Start(300)
        wx.CallLater(50, self.sliderReleased)                           # kurz verzögert, weil EVT_SLIDER nach EVT_SCROLL_CHANGED noch einmal kommt

    # ------------------------------------------------------------------
    # Wird kurz verzögert aus sliderUpdateDone() aufgerufen.
    def sliderReleased(self):
        self.slider_grabbed=False

    # ------------------------------------------------------------------
    # Verzeichnisauswahl-Dialog öffnen, nach Auswahl die map-Dateien
    # daraus einlesen und mplayer sowie wxOSM starten.
    def sel_folder(self, event):
        dc=wx.DirDialog(self, "select folder", "~")
        if dc.ShowModal()==wx.ID_OK:
            self.folder=dc.GetPath().encode("utf-8")
            self.initAfterFolderSelection()
            self.play()

    # ------------------------------------------------------------------
    # Springt zur vorigen Video-Datei in der Playlist.
    def prev_file(self, event=None):
        if self.mplayer_process:
            self.cmd_lock.acquire()
            self.mplayer_process.stdin.write(b"key_down_event %d\n"%(ord("<"),))
            self.mplayer_process.stdin.flush()
            self.cmd_lock.release()

    # ------------------------------------------------------------------
    # Springt 30 Sekunden rückwärts in der Video-Datei.
    def minus_30(self, event):
        self.skip_secs(self.mplayer_cur_sec-30)

    # ------------------------------------------------------------------
    # Springt 5 Sekunden rückwärts in der Video-Datei.
    def minus_5(self, event):
        self.skip_secs(self.mplayer_cur_sec-5)

    # ------------------------------------------------------------------
    # Spielt die Playlist ab.
    def play(self, event=None):
        print("play")
        self.open_mplayer()
        self.osmWin=MapFrame(self, (54.805060, 9.524878), 15, (2, 2), "wxOSMmini", pos=(0, 0), size=(630, 750))
        self.track, self.track_dict=self.buildTrackFromMapData()
        self.osmWin.loadTrack(self.track)
        self.osmWin.centerPos(self.track[0][0][0], self.track[0][0][1])
        self.osmWin.Show()
        self.tour_dist_km=float(self.track[-1][1])

    # ------------------------------------------------------------------
    # Toggled den Pause-Modus von mplayer.
    def pause(self, event):
        if not self.mplayer_pause:
            self.pause_on()
        else:
            self.pause_off()

    # ------------------------------------------------------------------
    # Aktiviert den Pause-Modus von mplayer.
    def pause_on(self):
        print("pause an")
        self.mplayer_pause=not self.mplayer_pause
        self.timer.Stop()
        self.b_prev_file.Disable()
        self.b_minus_30.Disable()
        self.b_minus_5.Disable()
        self.b_plus_5.Disable()
        self.b_plus_30.Disable()
        self.b_next_file.Disable()
        if self.mplayer_process:
            self.cmd_lock.acquire()
            self.mplayer_process.stdin.write(b"pause\n")
            self.mplayer_process.stdin.flush()
            self.cmd_lock.release()
        
    # ------------------------------------------------------------------
    # Deaktiviert den Pause-Modus von mplayer.
    def pause_off(self):
        print("pause aus")
        self.mplayer_pause=not self.mplayer_pause
        self.b_prev_file.Enable()
        self.b_minus_30.Enable()
        self.b_minus_5.Enable()
        self.b_plus_5.Enable()
        self.b_plus_30.Enable()
        self.b_next_file.Enable()
        if self.mplayer_process:
            self.cmd_lock.acquire()
            self.mplayer_process.stdin.write(b"pause\n")
            self.mplayer_process.stdin.flush()
            self.cmd_lock.release()
        self.timer.Start(300)

    # ------------------------------------------------------------------
    # Springt "secs" Sekunden vorwärts in der Video-Datei.
    def skip_secs(self, secs):
        if self.mplayer_process and self.mplayer_cur_sec and self.mplayer_file_length:
            if self.mplayer_file_length<=self.mplayer_cur_sec:
                return
            self.cmd_lock.acquire()
            self.mplayer_process.stdin.write(b"seek %d 2\n"%(secs,))
            self.mplayer_process.stdin.flush()
            self.mplayer_cur_sec=None
            self.cmd_lock.release()

    # ------------------------------------------------------------------
    # Springt 5 Sekunden vorwärts in der Video-Datei.
    def plus_5(self, event):
        self.skip_secs(self.mplayer_cur_sec+5)

    # ------------------------------------------------------------------
    # Springt 30 Sekunden vorwärts in der Video-Datei.
    def plus_30(self, event):
        self.skip_secs(self.mplayer_cur_sec+30)

    # ------------------------------------------------------------------
    # Springt zur nächsten Video-Datei in der Playlist.
    def next_file(self, event=None):
        if self.mplayer_process:
            self.cmd_lock.acquire()
            self.mplayer_process.stdin.write(b"key_down_event %d\n"%(ord(">"),))
            self.mplayer_process.stdin.flush()
            self.cmd_lock.release()

    # ------------------------------------------------------------------
    # Vor dem Selbstmord erstmal die Subfenster schließen.
    def onClose(self, event):
        self.timer.Stop()
        if self.mplayer_process:
            self.mplayer_process.stdin.write(b"quit\n")
            self.mplayer_process.stdin.flush()
        if self.osmWin:
            self.osmWin.Destroy()
        time.sleep(0.1)
        self.parent.Destroy()

    # ------------------------------------------------------------------
    # Öffnet den mplayer im Slave-Mode und lädt die Playlist.
    # Weiterhin wird ein Thread aufgesetzt, der Antworten des mplayers
    # annimmt und in Queues verteilt.
    def open_mplayer(self):
        cmd=["/usr/bin/mplayer", "-playlist", self.playlist_name, "-slave", "-quiet", "-x", "1280", "-y", "720", "-geometry", "630:0" ]
        self.cmd_resp_queue=Queue.Queue()
        self.cur_fn_queue=Queue.Queue()
        self.mplayer_process=subprocess.Popen(cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE)
        worker=threading.Thread(target=self.pipe_print, name="pipe-print", args=(self.mplayer_process.stdout, self.cmd_resp_queue, self.cur_fn_queue))
        worker.setDaemon(True)
        worker.start()
        self.timer.Start(300)

    # ------------------------------------------------------------------
    # Status beim mplayer anfragen, Werte merken und wxOSM nachführen.
    def OnTimer(self, event):
        while not self.cur_fn_queue.empty():                            # Queues leer machen
            self.cur_fn_queue.get_nowait()
        while not self.cmd_resp_queue.empty():
            self.cmd_resp_queue.get_nowait()
        self.cmd_lock.acquire()
        try:
            self.mplayer_process.stdin.write(b"get_file_name\n")
        except IOError:
            self.cmd_lock.release()
            return
        #print("OnTimer", self.cur_fn_queue.qsize(), self.cmd_resp_queue.qsize())
        self.mplayer_process.stdin.flush()
        self.mplayer_filename=self.__get_filename_answer(self.cur_fn_queue)

        self.mplayer_process.stdin.write(b"get_time_length\n")
        self.mplayer_process.stdin.flush()
        self.mplayer_file_length=self.__get_float_answer(self.cmd_resp_queue)

        self.mplayer_process.stdin.write(b"get_time_pos\n")
        self.mplayer_process.stdin.flush()
        self.mplayer_cur_sec=self.__get_float_answer(self.cmd_resp_queue)
        self.cmd_lock.release()
        try:
            p=int(round(self.mplayer_cur_sec, 0))+self.delta_ctrl.GetValue()
        except:
            print("keine Zeit empfangen")
            return                                                      # keine Zeit empfangen
        if p==self.mplayer_prev_sec:
            return                                                      # keine neue Zeit empfangen
        self.mplayer_prev_sec=p

        if str(self.mplayer_filename) in self.map_data:
            if p in self.map_data[self.mplayer_filename].line_dict:     # Zeit in map-Daten suchen
                self.cur_video_data=self.map_data[self.mplayer_filename].line_dict[p]
                print(self.cur_video_data, "%f %f"%(self.cur_video_data["lat"], self.cur_video_data["lon"])   )
                self.osmWin.centerPos(self.cur_video_data["lat"], self.cur_video_data["lon"])
            else:
                self.cur_video_data=None
                print("not in map")
            self.cur_file_index_in_playlist=(self.mp4_files.index(self.mplayer_filename)+1, len(self.mp4_files))
            self.updateFileData()
            self.updateTourData()

    # ------------------------------------------------------------------
    # Läuft als Thread.
    # Liest aus der stdout-pipe vom mplayer, prüft auf bekannte
    # Antwort-Präfixe und schreibt die Antworten in Queues je
    # Antwort-Typ - bzw. stoppt den Timer, wenn mplayer shutdown meldet.
    def pipe_print(self, pipe, cmd_resp_queue, cur_fn_queue):
        for line in iter(pipe.readline, b''):
            ln=line.decode("utf-8").strip()
            #print(">>>>", ln)
            if ln.startswith("ANS_LENGTH") or ln.startswith("ANS_TIME_POSITION"):
                cmd_resp_queue.put(ln)
            elif ln.startswith("ANS_FILENAME"):
                cur_fn_queue.put(ln)
            elif ln.startswith("Exiting... (Quit)"):
                self.mplayer_process=None
                print("mplayer geschlossen")
                self.timer.Stop()
            elif ln.startswith("Exiting... (End of file)"):
                self.mplayer_process=None
                print("mplayer am Ende der Playlist")
                self.timer.Stop()

    # ------------------------------------------------------------------
    # Liefert eine float-Antwort aus der Queue "q".
    def __get_float_answer(self, q):
        try:
            kv=q.get(timeout=.1).split("=")
            return float(kv[1])
        except:
            return None

    # ------------------------------------------------------------------
    # Liefert eine string-Antwort aus der Queue "q".
    def __get_filename_answer(self, q):
        try:
            kv=q.get(timeout=.1).split("=", 1)
            return kv[1].strip("'")
        except:
            return None


# ----------------------------------------------------------------------
# 
class Z3dOsmFrame(wx.Frame):
    def __init__(self, pos, size):
        style=wx.DEFAULT_FRAME_STYLE
        nam="wxZ3D-OSM v%s"%(VERSION,)
        wx.Frame.__init__(self, None, wx.ID_ANY, nam, pos=pos, size=size, style=style)
        if pos==wx.DefaultPosition:
            self.Centre()
        Z3dOsmPanel(self)


# ----------------------------------------------------------------------
# 
if __name__=="__main__":
    app=wx.App()
    frame=Z3dOsmFrame((1200, 750), (690, 370))
    frame.Show()
    app.MainLoop()
