#! /usr/bin/env python
# -*- coding: utf-8 -

# map_parse.py
#
# Klassen zur Verarbeitung von map- und mp4-Dateien in einem Verzeichnis.
#
# FilesInFolder     : Operationen auf einem gesamten Verzeichnis
# SingleFileData    : Datenstruktur für eine map-Datei mit ihren validen Daten
# ParseMap          : Operationen auf map-Dateien
#
# D.Ahlgrimm    13.02.2022

import os
import sys
import datetime
import time
import argparse

from geo_converter import GeoConverter

# ----------------------------------------------------------------------
# Kapselt die Datei-Operationen auf einem Ordner mit map- und
# mp4-Dateien. Diese Dateien müssen immer paarig auftreten.
class FilesInFolder():
    def __init__(self):
       pass

    # ------------------------------------------------------------------
    # Lädt alle map und mp4-Dateinamen aus "folder" und liefert sie in
    # zwei sortierten Listen zurück.
    # Der hier genutzte Algorithmus funktioniert nur dann, wenn zu
    # jeder mp4-Datei eine map-Datei existiert, deren Dateiname
    # sortiertechnisch unmittelbar vor der mp4-Datei liegt.
    def getNames(self, folder):
        map_files=[]
        mp4_files=[]
        for curdir, subdirlst, filenamelst in os.walk(folder):
            map_file=mp4_file=""
            for fn in sorted(filenamelst):
                if fn.lower().endswith(".map"):
                    map_file=fn
                if fn.lower().endswith(".mp4"):
                    mp4_file=fn
                if map_file!="" and mp4_file!="":
                    map_files.append(map_file)
                    mp4_files.append(mp4_file)
                    map_file=mp4_file=""
        return map_files, mp4_files

    # ------------------------------------------------------------------
    # Liefert alle mp4-Dateinamen aus "folder" mit Pfadangabe als Liste.
    def getMP4NamesWithPath(self, folder):
        map_files, mp4_files=self.getNames(folder)
        res=[]
        for mp4 in mp4_files:
            res.append(os.path.join(folder, mp4))
        return res

    # ------------------------------------------------------------------
    # Lädt alle map und mp4-Dateinamen aus "folder" und liefert sie in
    # in einem Dict zurück. Key ist die map-Datei, Value die mp4-Datei.
    # Beide jeweils ohne Pfad.
    def getNamePairList(self, folder):
        files=[]
        for curdir, subdirlst, filenamelst in os.walk(folder):
            map_file=mp4_file=""
            for fn in sorted(filenamelst):
                if fn.lower().endswith(".map"):
                    map_file=fn
                if fn.lower().endswith(".mp4"):
                    mp4_file=fn
                if map_file!="" and mp4_file!="":
                    files.append({"map": map_file, "mp4": mp4_file})
                    map_file=mp4_file=""
        return files

    # ------------------------------------------------------------------
    # Erstellt eine mplayer-Playlist für die mp4-Dateien in "folder"
    # und speichert die Playlist als "playlist_name".
    def buildPlaylist(self, folder, playlist_name):
        map_files, mp4_files=self.getNames(folder)
        with open(playlist_name, 'wt') as fl:
            for mp4 in mp4_files:
                fn=os.path.join(folder, mp4)
                fl.write(fn+"\n")



# ----------------------------------------------------------------------
# Eine Datenstruktur für ein Pärchen aus mp4- und map-Datei mit den
# validen Zeilen aus der map-Datei im Format
#   { zeilennummer : {"lat": lat, "lon": lon, "spd", spd, "dt": dt },
#     zeilennummer : {"lat": lat, "lon": lon, "spd", spd, "dt": dt },
#     [....]
#   }
class SingleFileData():
    def __init__(self, name_dict, line_dict):
        self.map_filename=name_dict["map"]
        self.mp4_filename=name_dict["mp4"]
        self.line_dict=line_dict



# ----------------------------------------------------------------------
# Funktion[en] zur Auswertung einer map-Datei im Verzeichnis "folder"
# gemäß __init__().
class ParseMap():
    # ------------------------------------------------------------------
    # Initialisiert diese Klasse für die map-Dateien in "folder".
    def __init__(self, folder):
        self.gc=GeoConverter()
        self.folder=folder
        self.first_valid_timestamp=None                                 # wird in parseAll() auf die erste gültige Zeit der Tour gesetzt
        self.last_valid_timestamp=None                                  # wird in parseAll() auf die letzte gültige Zeit der Tour gesetzt
        self.tour_in_seconds=None                                       # wird in parseAll() auf Tour-Dauer [Sek] gesetzt

    # ------------------------------------------------------------------
    # Liefert den validen Inhalt der map+mp4-Datei "name_dict" als
    # SingleFileData.
    def parseSingle(self, name_dict):
        with open(os.path.join(self.folder, name_dict["map"]), 'r') as fl:
            lines=fl.readlines()
        line_number=1
        last_dt=last_tm=""
        valid_lines=dict()
        for real_line_number, line in enumerate(lines):
            line=line.strip()
            elems=line.split(",")
            #mrk, dt, tm, lat, ns, lon, we, spd, *_=elems               # Python3
            mrk, dt, tm, lat, ns, lon, we, spd=elems[:8]                # Python2
            if mrk=="V":                                                # Zeile ohne GPS-Infos
                if last_dt==dt and last_tm==tm:                         # bei Zeit-Dupes...
                    #print("dupe:", dt, tm)
                    continue                                            # ...Zeile nicht zählen
                line_number+=1
                last_dt, last_tm=dt, tm
            elif mrk=="A":                                              # gültige Zeile
                dd, mm, yy=dt[0:2], dt[2:4], dt[4:6]
                hh, mn, ss=tm[0:2], tm[2:4], tm[4:6]
                d=datetime.datetime(int(yy)+2000, int(mm), int(dd), int(hh), int(mn), int(ss))
                #du=datetime.datetime.fromtimestamp(time.mktime(time.gmtime(time.mktime(d.timetuple()))))
                #dt=du.strftime("%Y-%m-%dT%H:%M:%SZ")
                dt=d.strftime("%H:%M:%S %d.%m.%Y")
                valid_lines.update({line_number: {  "lat": self.gc.ddm2dd(lat, ns), 
                                                    "lon": self.gc.ddm2dd(lon, we),
                                                    "spd": spd,
                                                    "dt":  dt    }})
                line_number+=1
            else:
                print("ERROR: illegal first element in line %s"%(real_line_number+1,))
        return SingleFileData(name_dict, valid_lines)

    # ------------------------------------------------------------------
    # Liefert alle Daten aus dem Verzeichnis gemäß __init__() als Dict
    # im Format
    #   {   mp4-file: SingleFileData,
    #       mp4-file: SingleFileData,
    #       [...]
    #   }
    # Auch werden getTourTimes() und getFileForSec() mit Daten beschickt.
    def parseAll(self):
        folderOps=FilesInFolder()
        files=folderOps.getNamePairList(self.folder)                    # Format: [ {"map": map_filename, "mp4": mp4_filename}, ... ]
        result_dict={}
        times_list=[]
        for name_dict in files:                                         # sortiert über alle Dateien
            sfd=self.parseSingle(name_dict)
            result_dict.update({name_dict["mp4"]: sfd})
            times_list.append((name_dict["mp4"], name_dict["map"], sfd.line_dict[min(sfd.line_dict.keys())]["dt"], sfd.line_dict[max(sfd.line_dict.keys())]["dt"]))

        first_idx=min(result_dict[files[0]["mp4"]].line_dict)
        first_valid_timestamp=result_dict[files[0]["mp4"]].line_dict[first_idx]["dt"]
        self.first_valid_timestamp=datetime.datetime.strptime(first_valid_timestamp, "%H:%M:%S %d.%m.%Y")
        last_idx=max(result_dict[files[-1]["mp4"]].line_dict)
        last_valid_timestamp=result_dict[files[-1]["mp4"]].line_dict[last_idx]["dt"]
        self.last_valid_timestamp=datetime.datetime.strptime(last_valid_timestamp, "%H:%M:%S %d.%m.%Y")
        self.tour_in_seconds=(self.last_valid_timestamp-self.first_valid_timestamp).total_seconds()

        self.file_secs=[]
        for mp4_fn, map_fn, start_str, end_str in times_list:
            start_ts=datetime.datetime.strptime(start_str, "%H:%M:%S %d.%m.%Y")
            end_ts=datetime.datetime.strptime(end_str, "%H:%M:%S %d.%m.%Y")
            self.file_secs.append((mp4_fn, map_fn, int((start_ts-self.first_valid_timestamp).total_seconds()), int((end_ts-self.first_valid_timestamp).total_seconds())))
        return result_dict

    # ------------------------------------------------------------------
    # Liefert Start, Ende und Dauer der Tour. parseAll() muss gelaufen
    # sein.
    def getTourTimes(self):
        return self.first_valid_timestamp, self.last_valid_timestamp, self.tour_in_seconds

    # ------------------------------------------------------------------
    # Liefert den Dateinamen, innerhalb dessen Videodatei sich "sec"
    # befindet. parseAll() muss gelaufen sein.
    def getFileForSec(self, sec):
        for mp4_fn, map_fn, start_sec, end_sec in self.file_secs:
            if start_sec<=sec<=end_sec:                                 # die genau passende Datei finden
                return mp4_fn, map_fn, start_sec, end_sec
        for mp4_fn, map_fn, start_sec, end_sec in self.file_secs:       # wenn keine gefunden wurde...
            if start_sec>=sec:                                          # die letzte Position davor liefern
                return mp4_fn, map_fn, start_sec, end_sec

# ----------------------------------------------------------------------
# Setzt den ArgParser auf.
def setupArgParser():
    desc="%(prog)s - bla"
    usage='%(prog)s [-h] [-f FOLDER]'
    epilog='Examples:\n' \
         '  %(prog)s\n' \
         '     blub\n'

    parser=argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter,
                                    description=desc, usage=usage, epilog=epilog)
    parser.add_argument('-f', dest='folder', type=str,
                help='source directory (default is ".")')
    args=parser.parse_args()

    if args.folder and not os.path.isdir(args.folder):
        parser.error("folder not an existing directory!")

    if args.folder:
        args.folder=os.path.normpath(args.folder)
    else:
        args.folder=os.getcwd()
    return(args)

# ----------------------------------------------------------------------
# Tests
if __name__=="__main__":
    args=setupArgParser()

    mapParser=ParseMap(args.folder)
    res=mapParser.parseAll()                                            # Format: [ SingleFileData, SingleFileData, ... ]
    print(mapParser.getFileForSec(500))
    
    sys.exit()
    for fn, r in res.items():
        if 80 in r.line_dict:
            print(fn, r.map_filename, r.line_dict[100])
        else:
            print(fn, "---")
    sys.exit()


    folderOps=FilesInFolder()
    
    #folderOps.buildPlaylist(args.folder, "/tmp/playlist.txt")
    #sys.exit()

    #map_files, mp4_files=folderOps.getNames(args.folder)
    #print(map_files)
    #print()
    #print(mp4_files)

    files=folderOps.getNamePairList(args.folder)
    #print(files)
    #sys.exit()

    mapParser=ParseMap(args.folder)

    map_data=mapParser.parseSingle(files[2])
    print(map_data.mp4_filename)
    print(map_data.map_filename)
    for k, v in map_data.map_dict.items():
        print(k, v)
        #print(ln, "%02s:%02d"%(ln[0]//60, ln[0]%60))
