#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Eine Klasse zum Laden und cachen von OpenStreetMap-Tiles.
#
# Detlev Ahlgrimm, 2017
#
# 08.04.2017  erste Version
# 10.04.2017  Umstellung auf asynchrones Laden der Tiles
# 14.04.2017  self.status eingebaut, damit eine Unterscheidung zwischen
#             "laden unmöglich" und "laden fehlgeschlagen" erfolgen kann.
# 23.04.2017  "force_reload" und __removeTile() zugefügt
# 24.04.2017  RAMcache zugefügt, Löschbedingung für self.already_requested erweitert
# 06.09.2017  Option NO_TIMEOUT zugefügt (Tiles, die in Timeout gelaufen sind, werden gesondert nach-requestiert)
# 07.09.2017  Option OFFLINE zugefügt
#


from PIL import Image   # python-Pillow-2.8.1-3.6.1.x86_64 
import requests         # python-requests-2.4.1-3.1.noarch
from io import BytesIO
import os
import time
import datetime
import threading
from Queue import Queue

DEBUG=True
CACHE_MAX_TILES=1000      # Maximal-Anzahl für RAMCache
MAX_TILE_AGE_DAYS=360      # maximales Alter von Tiles - ältere Tiles neu laden
#NO_TIMEOUT=False          # bei True wird kein Timeout gemeldet - sondern intern "gequeued"
NO_TIMEOUT=True
#OFFLINE=True              # bei True wird alles aus dem HDD-Cache bedient
OFFLINE=False

class TileCache():
  def __init__(self, cache_base_dir, tile_url):
    self.cache_base_dir=cache_base_dir
    if self.cache_base_dir is None:
      self.cacheRAM=RAMcache()
    self.tile_url=tile_url
    self.queue=Queue()      # Auftrags-Übergabe für die Threads
    self.running_count=0    # wird pro arbeitendem Thread hochgezählt und wieder runter, wenn er fertig ist
    self.running_lock=threading.Lock()  # ein Lock() für Schreibzugriffe auf "self.running_count"
    self.num_worker_threads=64    # Anzahl Worker-Threads
    self.request_timeout=5        # Maximal Wartezeit in Sekunden, bevor Ladevorgang abgebrochen wird
    self.already_requested=list() # Liste aller aktuell zu holenden Tiles
    self.status={"GOOD":0, "WAIT":1, "FAILED":2, "INVALID":3}
    self.reload_later=list()
    for i in range(self.num_worker_threads):
      worker=threading.Thread(target=self.__getTileAsyncFromWeb, name="getTileAsyncFromWeb("+str(i)+")")
      worker.setDaemon(True)
      worker.start()
    if NO_TIMEOUT:
      worker=threading.Thread(target=self.__delayedReload, name="delayedReload()")
      worker.setDaemon(True)
      worker.start()
      

  # ######################################################################
  # Liefert das OSM-Tile(x, y, z) mit Status als Image oder None.
  def getTileImg(self, x, y, z, force_reload=False):
    if OFFLINE:
      fldr=os.path.join(self.cache_base_dir, str(z), str(x))
      fl=os.path.join(fldr, "%d.png"%(y,))
      try:
        img=Image.open(fl)
        image=img.convert('RGB')
      except:
        image=None
      return((self.status["GOOD"], image))
    if self.queue.empty() and self.running_count==0:  # wenn queue leer ist und alle Threads idlen...
      self.already_requested=list()                   # ...können die alten Merker weg
    if self.cache_base_dir is not None:           # wenn im Dateisystem gecached werden soll
      fldr=os.path.join(self.cache_base_dir, str(z), str(x))  # Pfad bauen
      fl=os.path.join(fldr, "%d.png"%(y,))                    # Tile-Name samt Pfad bauen
      if force_reload:          # wenn "reload" angefordert wurde...
        self.__removeTile(fl)   # Tile vorsichtshalber löschen, damit es auch trotz Timeout sicher "reloaded" wird
      if self.__isOnDisk(fl) and not force_reload:  # wenn Tile schon im Cache existiert und nicht reload angefordert wurde
        try:
          img=Image.open(fl)                        # Tile aus Cache liefern
        except:
          return((self.status["FAILED"], None))     # bei Fehler später nochmal probieren
      else:
        if (x, y, z) in self.reload_later:          # wenn Tile zuvor nicht aus dem Web geladen werden konnte
          return((self.status["GOOD"], None))       # Leer-Tile ohne Nachlieferungs-Meldung zurückgeben
        if (x, y, z) not in self.already_requested: # wenn Anfrage für Tile noch nicht offen ist
          if x<0 or x>=2**z or y<0 or y>=2**z:      # wenn Tile nicht existiert bzw. illegal ist
            # http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Zoom_levels
            return((self.status["INVALID"], None))  # als ungültiges Tile markieren
          self.queue.put((x, y, z))                 # asynchrones Laden des Tiles beauftragen
          self.already_requested.append((x, y, z))  # und Auftrag merken, um ihn nur einmal abzusetzen
        return((self.status["WAIT"], None))         # Nachlieferungs-Meldung
    else:                                         # wenn im RAM gecached werden soll
      img=self.cacheRAM.getFromRAM(x, y, z)       # alles analog zu oben
      if img is not None and not force_reload:
        pass
      else:
        if (x, y, z) not in self.already_requested:
          if x<0 or x>=2**z or y<0 or y>=2**z:
            return((self.status["INVALID"], None))
          self.queue.put((x, y, z))
          self.already_requested.append((x, y, z))
        return((self.status["WAIT"], None))
    try:
      image=img.convert('RGB')                    # den convert() braucht der wx.BitmapFromBuffer()
    except:
      return((self.status["FAILED"], None))       # wenn Fehlschlag - später nochmal probieren
    return((self.status["GOOD"], image))          # und zurückliefern


  # ######################################################################
  #
  def __delayedReload(self):
    while True:
      time.sleep(0.1)
      if len(self.reload_later)>0 and self.queue.empty(): # wenn Tiles auf Reload warten und die Queue leer ist
        x, y, z=self.reload_later.pop(0)                  # das älteste Tile
        print "reload(%d,%d,%d)\n"%(x, y, z),
        self.queue.put((x, y, z))                         # auf die Queue
        self.already_requested.append((x, y, z))


  # ######################################################################
  # Liefert True, wenn "filename" auf HDD existiert und jünger als
  # MAX_TILE_AGE_DAYS ist.
  def __isOnDisk(self, filename):
    if os.path.exists(filename):
      mtime=os.path.getmtime(filename)
      last_modified_date=datetime.datetime.fromtimestamp(mtime)
      age=datetime.datetime.now()-datetime.datetime.fromtimestamp(mtime)
      #if age.days>=MAX_TILE_AGE_DAYS: print age.days
      return(age.days<MAX_TILE_AGE_DAYS)
    return(False)


  # ######################################################################
  # Löscht das Tile "filename" aus dem Dateisystem.
  def __removeTile(self, filename):
    try:
      os.remove(filename)
    except:
      pass


  # ######################################################################
  # Löscht alle anstehenden Requests, die nicht schon von einem
  # Worker-Thread angenommen wurden.
  def flushQueue(self):
    while not self.queue.empty():
      self.queue.get()
    self.already_requested=list()


  # ######################################################################
  # Liefert die Anzahl der noch unbeantworteten Tile-Requests.
  def getQueueLength(self):
    return("%d / %d"%(self.queue.qsize()+self.running_count, len(self.reload_later)))


  # ######################################################################
  # Liefert die Textstrings und Fehlernummern, die bei Fehler statt
  # Bitmaps geliefert werden können als Dictionary.
  def getStatusDict(self):
    return(self.status)


  # ######################################################################
  # Worker-Thread, um ein OSM-Tile zu laden und lokal zu speichern.
  def __getTileAsyncFromWeb(self):
    while True:
      job=self.queue.get()
      self.__incRunningCount()
      x, y, z=job
      s, img=self.__getTileFromWeb(x, y, z)
      if s==self.status["GOOD"]:
        if self.cache_base_dir is None:
          self.cacheRAM.putIntoRAM(x, y, z, img)
        else:
          fldr=os.path.join(self.cache_base_dir, str(z), str(x))
          fl=os.path.join(fldr, "%d.png"%(y,))
          try:
            os.makedirs(fldr)
          except:
            pass  # war wohl schon existent == macht nix
          try:
            img.save(fl)
          except:
            if DEBUG: print "Error img.save()\n",
            pass
      elif NO_TIMEOUT:
        self.reload_later.append((x, y, z))
      self.queue.task_done()
      self.__decRunningCount()


  # ######################################################################
  # Der += bzw. -= scheint nicht "atomar" implementiert zu sein - daher
  # muss der Schreibzugriff per Lock() Thread-safe gemacht werden.
  def __incRunningCount(self):
    self.running_lock.acquire()
    self.running_count+=1
    self.running_lock.release()
  def __decRunningCount(self):
    self.running_lock.acquire()
    self.running_count-=1
    self.running_lock.release()


  # ######################################################################
  # Das OSM-Tile(x, y, z) aus dem Web laden und samt Status zurückliefern.
  def __getTileFromWeb(self, x, y, z):
    #if DEBUG: print "Web-request(%d,%d,%d)\n"%(x, y, z),
    url=self.tile_url.format(z=z, x=x, y=y)
    #print url
    try:
      response=requests.get(url, timeout=self.request_timeout)
    except:
      if DEBUG: print "Timeout(%d,%d,%d)\n"%(x, y, z),
      return((self.status["FAILED"], None))
    try:
      img=Image.open(BytesIO(response.content))
      if DEBUG: print "cached(%d,%d,%d)\n"%(x, y, z),
    except:
      if DEBUG: print "Error in Image.open()\n",
      return((self.status["FAILED"], None))
    return((self.status["GOOD"], img))



# ######################################################################
# Ein kleiner Cache im RAM.
# Ganz ohne Cache kann OSMscr2 nicht arbeiten, weil mögliche
# Tile-Nachlieferungen ja irgendwo hin müssen.
# Für einen lokalen Tile-Server könnte natürlich auch komplett auf
# synchrones Laden der Tiles gewechselt werden.... also quasi so, dass
# getTile() direkt und nur __getTileFromWeb() aufruft.
class RAMcache():
  def __init__(self):
    self.ram_cache=dict()
    self.max_tiles=CACHE_MAX_TILES
    self.age=0
    self.oldest=0
    self.lock=threading.Lock()  # Schreibzugriffe auf self.ram_cache serialisieren


  # ######################################################################
  # Legt das Tile(x, y, z) im RAM ab und löscht bei Erreichen der
  # Tile-Maximal-Anzahl die Tiles, die am längsten nicht angefasst wurden.
  def putIntoRAM(self, x, y, z, tile):
    self.lock.acquire()
    lv=len(self.ram_cache)
    if len(self.ram_cache)>self.max_tiles:
      del_cnt=(self.age-self.oldest)/10     # die ältesten 10% löschen - naja grob 10%
      self.oldest+=del_cnt
      del_lst=list()
      for xyz, (dummy, age) in self.ram_cache.items():
        if age<self.oldest:
          del_lst.append(xyz)
      for xyz in del_lst:
        del self.ram_cache[xyz]
    self.ram_cache.update({(x, y, z):(tile, self.age)})
    self.age+=1
#    print "len:%3d-%3d  age:%5d  oldest:%5d\n"%(lv, len(self.ram_cache), self.age, self.oldest),
    self.lock.release()


  # ######################################################################
  # Liefert das Tile(x, y, z) aus dem RAM - oder None, wenn es dort noch
  # nicht existiert.
  def getFromRAM(self, x, y, z):
    tile, age=self.ram_cache.get((x, y, z), (None, None))
    if tile is not None:
      self.lock.acquire()
      self.ram_cache.update({(x, y, z):(tile, self.age)})
      self.age+=1
      self.lock.release()
    return(tile)
