#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Eine Klasse zur Anzeige einer OpenStreetMap-Karte samt
# Scroll- und Zoom-Steuerung. Weiterhin können Tracks und
# Punkte verwaltet und dargestellt werden.
#
# Detlev Ahlgrimm, 2017
#
#
# 08.04.2017  erste Version
# 10.04.2017  kompletter Umbau, der jetzt die Position (0, 0) als Referenz-Punkt
#             verwendet (statt wie vorher den Mittelpunkt der Anzeige).
# 13.04.2017  nochmal deutlicher Umbau, weil ein statischer Faktor für "lat/lon zu Pixel"
#             nicht wirklich funktioniert - zumindest nicht für lat.
# 14.04.2017  nicht existente Tiles (außerhalb der Karte) erkennen und nicht
#             ständig versuchen, sie zu laden (bzw. Unterscheidung zwischen
#             "laden unmöglich" und "laden fehlgeschlagen").
# 15.04.2017  getVisibleRectangle() und getZoomAndCenterForAllPointsVisible() zugefügt.
# 16.04.2017  Umstellung auf Nutzung von wx.MemoryDC(),
#             Verarbeitung von Track- und Point-Listen.
# 17.04.2017  Verarbeitung mehrerer unabhängiger Point-Listen, die über ein Handle
#             referenziert werden, Tracks werden Ausschnitts-bedingt an DrawLines() übergeben,
#             nur solche Punkte an DrawCirclePoint() übergeben, die auf der Bitmap liegen.
# 29.04.2017  Verarbeitung von Track-Listen nun analog zu Point-Listen.
# 30.04.2017  __getVisibleTrackParts() zugefügt.
# 02.05.2017  self.pos0LL in die Änderungserkennung für Karten-Neuaufbau mit aufgenommen. Dadurch
#             konnte der "self.needs_refresh=True" aus __setPoint() raus und dadurch geht die
#             CPU-Last deutlich runter, wenn (bei Autofollow) ständig die selbe Position gefordert wird.
# 14.05.2017  Umstellung auf OSMpoints().
# 03.06.2017  wxPython komplett rausgeschmissen und stattdessen Nutzung von pillow,
#             Tracks und Point-Listen werden auf einzelne Tiles geschrieben (statt wie vorher aufs Ziel-Bitmap).
# 04.06.2017  selektives Tile-Löschen bei unloadTrack() und unloadPoints().
# 06.06.1017  Ausblenden von übereinader liegenden Points in __drawOnTile().
# 07.06.2017  dictTN wird per dictTN_z zum Zoom-Level gecached.
# 08.06.2017  Umstellung auf OSMtile(), um nur dann neu auf Tiles zeichen zu müssen, wenn tatsächlich
#             eine Änderung der darauf abgebildeten Tracks/Points stattgefunden hat.
# 14.06.2017  Erstellung von getNewMapPointsForTrack().
# 16.06.2017  Erstellung von getTrackPartAsPixelDelta().
# 17.06.2017  Umstellung von getTrackPartAsPixelDelta() - jetzt werden die Linien alle relativ zum Startpunkt
#             berechnet und nicht mehr relativ zum Vorgänger-Punkt (da summierten sich Fehler von Pixelbruchteilen).
# 20.06.2017  Erstellung von findTrackForPoint() und findTrackPointIndexForPoint().
# 27.06.2017  Umstellung des Trackformats von ((lat, lon), timestamp) auf ((lat, lon), data)
#

import time
import math
import copy
from PIL import Image, ImageDraw, ImageColor
import threading


from OSMtilecache import TileCache


# ######################################################################
# Eine Datenstruktur zur Aufnahme der Ausschnitts-relevanten Variablen
# von OSMscr3().
class OSMscr3Data():
  def __init__(self):
    self.pos0LL=None            # die Geo-Koordinaten der (0, 0)-Position im Fenster
    self.tile0TN=None           # die TileNummer vom Tile unter der (0, 0)-Position im Fenster
    self.needs_refresh=True     # die Kennung, dass im Bitmap noch Leer-Tiles enthalten sind
    self.shiftXY=(0, 0)         # der aktuelle Anzeige-Versatz vom aktuellen Bitmap bezogen auf die (0, 0)-Position im Fenster
    self.old_params=None        # das Variablen-Tupel zur Änderungserkennung
    self.border_fixXY=(0, 0)    # der Pixel-Versatz zwischen Fenster(0, 0) und der linken oberen Ecke von Tile[0][0]
    self.thread=None            # Zeiger auf den ggf. noch laufenden Thread
    self.wx_bitmap=None         # die Parameter (width, height, dataBuffer) für wx.BitmapFromBuffer()

# ######################################################################
# Eine Datenstruktur für einen einzelnen Track.
class OSMtrack():
  def __init__(self):
    self.listLL=None            # der Track als Liste von Tupeln (lat/lon-Wert, data)
    self.dictTN=None            # der Track als Dictionary zu Tile-Nummern mit Listen von Track-Punkten
    self.valid_for_zoom=-1      # der Zoom-Level, für den dictTN gebildet wurde
    self.color=(199, 21, 133)   # die Farbe, mit der der Track gezeichnet werden soll
    self.width=2                # die Linienstärke, mit der der Track gezeichnet werden soll
    self.dictTN_z=dict()        # dictTN zu Zoom-Level {zoom:dictTN}
    self.newMapPoints=dict()    # newMapPoints zu Zoom-Level {zoom:newMapPoints}

# ######################################################################
# Eine Datenstruktur für eine einzelne Point-Liste.
class OSMpoints():
  def __init__(self):
    self.listLL=None            # die Punkte als Liste von Tupeln (lat/lon-Wert, radius, farbe, data)
    self.dictTN=None            # die Punkte als Dictionary zu Tile-Nummern mit Listen von Punkten
    self.valid_for_zoom=-1      # der Zoom-Level, für den dictTN gebildet wurde
    self.dictTN_z=dict()        # dictTN zu Zoom-Level {zoom:dictTN}

# ######################################################################
# Eine Datenstruktur für ein Tile mit Tracks/Points.
# Beim Laden von Tracks/Points muss "img" erweitert werden.
# Beim Löschen von Tracks/Points müssen alle OSMtile's geprüft werden,
# ob das Track/Points-Handle in has_tracks/has_points vorkommt. In
# diesem Fall ist das OSMtile zu löschen.
class OSMtile():
  def __init__(self):
    self.img=None           # das Tile mit Tracks und Points
    self.has_tracks=set()   # Set von Track-Handles, die auf "img" gezeichnet wurden
    self.has_points=set()   # Set von Point-Handles, die auf "img" gezeichnet wurden
    self.no_tracks=set()    # Set von Track-Handles, die nicht auf "img" liegen
    self.no_points=set()    # Set von Point-Handles, die nicht auf "img" liegen



# ######################################################################
# Die Referenz ist immer der Punkt (0, 0) im Fenster. Also links oben.
#
# lat - Y-Achse, bei uns gehen höhere Werte nach Norden
# lon - X-Achse, bei uns gehen höhere Werte nach Westen
# südlich vom Äquator wird lat negativ
# westlich von Greenwich wird lon negativ
#
# Notation für Variablennamen:
#   variableXY  - Koordinaten in Pixel
#   variableTN  - TileNumber
#   variableLL  - Koordinaten in Lat/Lon
class OSMscr3():
  def __init__(self, srv_data):
    self.ts=srv_data["tile_size"]
    self.tc=TileCache(srv_data["cache_dir"], srv_data["tile_url"])
    self.max_zoom=srv_data["max_zoom"]
    self.status=self.tc.getStatusDict()
    self.show_grid=False

    self.v=dict()             # die variablen Daten einer Instanz     Format: {handle:OSMscr3Data()}
    self.all_tracks=dict()    # alle Tracks.                          Format: {handle:OSMtrack()}
    self.all_points=dict()    # alle Point-Listen.                    Format: {handle:OSMpoints()}
    self.tiles=dict()         # Tile-Images mit Tracks oder Punkten.  Format: {(x, y, z):OSMtile()}


  # ######################################################################
  # Initialisiert die Basis-Instanz und liefert dessen Handle zurück.
  def setup(self, dc_sizeXY, centerLL, zoom, border=(2, 2)):
    self.zoom=zoom
    self.border=border
    self.setSize(dc_sizeXY)
    handle=0
    self.v={handle:OSMscr3Data()}
    self.setCenter(centerLL)
    return(handle)


  # ######################################################################
  # Erstellt eine weitere Instanz und liefert dessen Handle zurück.
  def newMap(self, posLL=None):
    handle=max(self.v)+1
    self.v.update({handle:OSMscr3Data()})
    if posLL is not None:
      self.setCenter(posLL, handle)
    return(handle)


  # ######################################################################
  # Löscht die Daten zur Instanz "handle".
  def dropMap(self, handle):
    if handle!=0 and handle in self.v:        # wenn Handle existiert und nicht das Haupt-Handle ist
      if self.v[handle].thread is not None:   # wenn auf diesem Handle noch gerechnet wird...
        self.v[handle].thread.join(0.5)       # ...warten, bis das durch ist
      del self.v[handle]                      # Daten zum Handle löschen


  # ######################################################################
  # Passt alle entsprechenden Variablen einer [neuen] Fenstergröße an.
  def setSize(self, dc_sizeXY):
    self.dc_sizeXY=dc_sizeXY
    # Anzahl der sichtbaren Tiles (x, y) je nach Anzeigegröße
    self.tile_cnt_dc=(float(dc_sizeXY[0])/self.ts, float(dc_sizeXY[1])/self.ts)
    self.tile_cnt_dc_max=(int(math.ceil(self.tile_cnt_dc[0])), int(math.ceil(self.tile_cnt_dc[1])))


  # ######################################################################
  # Richtet die Karte so aus, dass der Längen/Breitengrad "centerLL" in
  # der Mitte des Anzeigebereiches zu liegen kommt.
  def setCenter(self, centerLL, handle=0):
    self.__setPoint(centerLL, (self.dc_sizeXY[0]/2, self.dc_sizeXY[1]/2), handle)


  # ######################################################################
  # Richtet die Karte so aus, dass der Längen/Breitengrad "posLL" auf der
  # Pixel-Position "posXY" zu liegen kommt.
  def __setPoint(self, posLL, posXY, handle=0):
    posXYs=(posXY[0]+self.v[handle].shiftXY[0], posXY[1]+self.v[handle].shiftXY[1]) # aktuellen Map-Versatz berücksichtigen
    cpdTN=self.deg2numF(posLL, self.zoom)                                           # Tile-Nummer unter der Ziel-Koordinate (float)
    cpbTN=(int(cpdTN[0]), int(cpdTN[1]))                                            # Tile-Nummer unter der Ziel-Koordinate (int)
    cprXY=((cpdTN[0]-cpbTN[0])*self.ts, (cpdTN[1]-cpbTN[1])*self.ts)                # Pixel-Versatz zu cpdTN
    tcbpXY=(posXYs[0]-cprXY[0], posXYs[1]-cprXY[1])                                 # Basis-Position von Tile unter der Ziel-Koordinate
    tdist=(int(math.ceil(tcbpXY[0]/self.ts)), int(math.ceil(tcbpXY[1]/self.ts)))    # Anzahl Tiles bis zum Tile auf (0, 0)
    tn00TN=(cpbTN[0]-tdist[0], cpbTN[1]-tdist[1])                                   # Tile-Nummer des Tiles auf (0, 0)
    tn0dXY=(tcbpXY[0]-tdist[0]*self.ts, tcbpXY[1]-tdist[1]*self.ts)                 # Basis-Position von Tile auf (0, 0)
    t0d=(abs(tn0dXY[0])/self.ts, abs(tn0dXY[1])/self.ts)                            # Abstand von Basis-Position zu (0, 0) in Tile-Bruchteilen
    self.v[handle].pos0LL=self.num2deg((tn00TN[0]+t0d[0], tn00TN[1]+t0d[1]), self.zoom) # Geo-Koordinaten der (0, 0)-Position im Fenster
    self.v[handle].tile0TN=self.deg2num(self.v[handle].pos0LL, self.zoom)           # TileNummer vom Tile auf (0, 0)


  # ######################################################################
  # Ändert die Zoom-Stufe und sorgt dafür, dass die Koordinate wieder an
  # der Position des Mauszeigers landet, die vor dem Zoom dort war.
  def setZoom(self, zoom, posXY, handle=0):
    if self.zoom==zoom:   # wenn der Zoom-Level nicht geändert wurde
      return              # ist nix zu tun
    tmp=dict()
    for h in self.v:                            # über alle Handles
      if h==handle:                             # beim aktuellen Handle
        llm=self.getLatLonForPixel(posXY, h)    # Position von posXY beim alten Zoom-Level merken
        tmp.update({h:(llm, posXY)})
      else:                                     # bei allen anderen Handles
        if self.v[h].tile0TN is not None:       # ...die schon initalisiert sind
          p=(self.dc_sizeXY[0]/2, self.dc_sizeXY[1]/2)
          llm=self.getLatLonForPixel(p, h)      # Position des Fenstermittelpunktes beim alten Zoom-Level merken
          tmp.update({h:(llm, p)})
    self.zoom=zoom                              # Zoom-Level ändern
    for h in tmp:
      self.__setPoint(tmp[h][0], tmp[h][1], h)  # jeweils gemerkte Position wieder einstellen

    for handle, data in self.all_tracks.items():    # tracks.dictTN cachen
      if zoom in data.dictTN_z:
        data.dictTN=copy.copy(data.dictTN_z[zoom])
      else:
        data.dictTN=self.__buildDataForGPXTrack(data.listLL)
        data.dictTN_z.update({zoom:copy.copy(data.dictTN)})
      data.valid_for_zoom=zoom
    for handle, data in self.all_points.items():    # points.dictTN cachen
      if zoom in data.dictTN_z:
        data.dictTN=copy.copy(data.dictTN_z[zoom])
      else:
        data.dictTN=self.__buildDataForGPXPoints(data.listLL)
        data.dictTN_z.update({zoom:copy.copy(data.dictTN)})
      data.valid_for_zoom=zoom
    self.tc.flushQueue()                        # alte, ggf. noch anstehende Tile-Requests können jetzt weg


  # ######################################################################
  # Bewegt das an Position (0, 0) angezeigte Pixel (samt der restlichen
  # Karte) um "distXY" Pixel.
  def doMoveMap(self, distXY, handle=0):
    self.v[handle].shiftXY=distXY


  # ######################################################################
  # Stellt das derzeit an Position (0, 0) angezeigte Pixel als neue
  # Basis ein.
  def endMoveMap(self, handle=0):
    self.v[handle].pos0LL=self.getLatLonForPixel((0, 0), handle)
    self.v[handle].tile0TN=self.deg2num(self.v[handle].pos0LL, self.zoom)
    self.v[handle].shiftXY=(0, 0)


  # ######################################################################
  # Liefert die aktuelle Map-Verschiebung.
  def getShift(self, handle=0):
    return(self.v[handle].shiftXY)


  # ######################################################################
  # Liefert Längen/Breitengrad unter dem Pixel "posXY".
  def getLatLonForPixel(self, posXY, handle=0):
    posXYs=(posXY[0]+self.v[handle].shiftXY[0], posXY[1]+self.v[handle].shiftXY[1]) # aktuellen Map-Versatz berücksichtigen
    tile0iLL=self.num2deg(self.v[handle].tile0TN, self.zoom)                        # lat/lon des Eckpunktes vom Tile auf (0, 0)
    tile0d0XY=self.distanceLatLonToPixel(self.v[handle].pos0LL, tile0iLL)           # Abstand Eckpunkt zu (0, 0)
    distXY=(posXYs[0]-tile0d0XY[0], posXYs[1]-tile0d0XY[1])                         # Abstand Eckunkt vom Tile auf (0, 0) zu posXY
    tdist=(float(distXY[0])/self.ts, float(distXY[1])/self.ts)                      # Abstand von posXY zu (0, 0) in Tiles
    tdp=(self.v[handle].tile0TN[0]+tdist[0], self.v[handle].tile0TN[1]+tdist[1])    # TileNummer mit Nachkommastellen unter "posXY"
    tpLL=self.num2deg(tdp, self.zoom)                                               # lat/lon unter "posXY"
    return(tpLL)


  # ######################################################################
  # Liefert zu einem Längen/Breitengrad "posLL" die Pixel-Koordinaten.
  def getPixelForLatLon(self, posLL, handle=0):
    dx, dy=self.distanceLatLonToPixel(self.v[handle].pos0LL, posLL)
    return(int(dx)-self.v[handle].shiftXY[0], int(dy)-self.v[handle].shiftXY[1])


  # ######################################################################
  # Liefert die Größe der Bitmap entsprechend Fenstergröße und Rand.
  def __getBitmapSize(self):
    return((self.tile_cnt_dc_max[0]+2*self.border[0])*self.ts, 
           (self.tile_cnt_dc_max[1]+2*self.border[1])*self.ts)


  # ######################################################################
  # Liefert die aktuelle Verschiebung zum Handle, die die linke obere
  # Ecke von Tile[0][0] zur Fenster-Koordinate (0, 0) versetzt ist.
  def getBitmapShift(self, handle=0):
    return((-(self.v[handle].shiftXY[0]+self.v[handle].border_fixXY[0]), 
            -(self.v[handle].shiftXY[1]+self.v[handle].border_fixXY[1])))


  # ######################################################################
  # Erstellt bei Bedarf eine neue Bitmap zum Handle und liefert das Tupel
  # (True, Versatz), wenn eine neue Bitmap erstellt wurde. Ansonsten wird
  # (False, Versatz) geliefert.
  def drawMap(self, handle=0, force_reload=False):
    if self.v[handle].old_params!=(self.dc_sizeXY, self.zoom, self.border, self.v[handle].pos0LL) or self.v[handle].needs_refresh:
      if self.v[handle].thread is not None:   # wenn auf diesem Handle noch gerechnet wird...
        self.v[handle].thread.join(0.5)       # ...warten, bis das durch ist
      t=threading.Thread(target=self.__drawMap, name="__drawMap(%d)"%(handle,), args=(handle, force_reload))
      self.v[handle].thread=t
      t.setDaemon(True)
      t.start()
      return(True, self.getBitmapShift(handle))
    return(False, self.getBitmapShift(handle))   # es wurde kein neues Bitmap erstellt


  # ######################################################################
  # Läuft als Thread und erstellt eine neue Bitmap zum Handle.
  def __drawMap(self, handle=0, force_reload=False):
    img, self.v[handle].border_fixXY=self.__getMapImage(force_reload, handle)
    self.v[handle].old_params=(self.dc_sizeXY, self.zoom, self.border, self.v[handle].pos0LL)
    w, h=img.size
    #self.v[handle].wx_bitmap=(w, h, img.tostring())
    self.v[handle].wx_bitmap=(w, h, img.tobytes())


  # ######################################################################
  # Liefert den aktuellen Karten-Ausschnitt als Bitmap, den Ausgabe-Versatz
  # als Tupel und True, wenn mindestens ein Tile ein Leer-Tile ist.
  def getBitmap(self, handle=0):
    if self.v[handle].thread is not None:
      self.v[handle].thread.join()
      self.v[handle].thread=None
    return(self.v[handle].wx_bitmap, self.getBitmapShift(handle), self.v[handle].needs_refresh)


  # ######################################################################
  # Liefert das Map-Image samt dem Versatz zwischen Tile[0][0] und der
  # Position (0, 0) im Fenster.
  # Setzt "self.v[handle].needs_refresh".
  def __getMapImage(self, force_reload=False, handle=0):
    tile0iLL=self.num2deg((self.v[handle].tile0TN[0]-self.border[0], self.v[handle].tile0TN[1]-self.border[1]), self.zoom) # lat/lon vom unsichtbaren Tile[0][0]
    border_fixXY=self.distanceLatLonToPixel(tile0iLL, self.v[handle].pos0LL)  # XY-Abstand vom Tile[0][0] zur Koordinate unter (0, 0)
    imo=Image.new("RGB", self.__getBitmapSize(), (204, 204, 204))
    self.v[handle].needs_refresh=False      # erstmal den positiven Fall annehmen
    for y in range(-self.border[1], self.tile_cnt_dc_max[1]+self.border[1]):
      for x in range(-self.border[0], self.tile_cnt_dc_max[0]+self.border[0]):
        img, needs_refresh=self.__getTile(self.v[handle].tile0TN[0]+x, self.v[handle].tile0TN[1]+y, self.zoom, force_reload)
        if needs_refresh:
          self.v[handle].needs_refresh=needs_refresh  # Kennung für "noch Leer-Tiles enthalten" setzen
        if img is not None:
          imo.paste(img, (self.ts*(x+self.border[0]), self.ts*(y+self.border[1])))
    return(imo, border_fixXY)


  # ######################################################################
  # Liefert False, wenn die Handles aller geladenen Track- und Punkt-
  # Listen in has_tracks/has_points vorkommen.
  def __rebuildNeededForTile(self, x, y, z, t):
    for handle, data in self.all_tracks.items():    # über alle Tracks
      if (x, y) in data.dictTN:                     # wenn aktueller Track auf dem Tile liegt
        if handle not in t.has_tracks:              # wenn er nicht schon draufgezeichnet ist
          return(True)                              # dann gibts was zu tun
    for handle, data in self.all_points.items():    # über alle Point-Listen
      if (x, y) in data.dictTN:                     # wenn aktuelle Point-Liste auf dem Tile liegt
        if handle not in t.has_points:              # wenn er nicht schon draufgezeichnet ist
          return(True)                              # dann gibts was zu tun
    return(False)                                   # das Tile ist auf aktuellem Stand -> nix zu tun


  # ######################################################################
  # Liefert das Tile (x, y, z) als Image samt Track-Linien und Points.
  # Als zweiter Wert wird True geliefert, wenn das Tile noch nicht
  # vorliegt - bzw. erneut angefordert werden muss.
  def __getTile(self, x, y, z, force_reload=False):
    hndoi=False
    if not force_reload and (x, y, z) in self.tiles:      # wenn Tile schon vorliegt und nicht neu geladen werden soll
      t=self.tiles[(x, y, z)]
      if not self.__rebuildNeededForTile(x, y, z, t):     # wenn kein Update nötig ist
        if t.img is None:                                 # wenn auf Tile keine Tracks/Points gezeichnet wurden
          hndoi=True                                      # merken, dass das nackige Tile geliefert werden soll
        else:                                             # ansonsten...
          return(t.img, False)                            # ...kann das Tile aus dem Cache geliefert werden
    else:                                                 # Tile liegt noch nicht vor
      t=OSMtile()                                         # Speicherplatz anlegen
      self.tiles.update({(x, y, z):t})                    # und (Pointer darauf) speichern
    status, im=self.tc.getTileImg(x, y, z, force_reload)  # Tile aus dem Web oder dem HDD-Cache holen
    if status!=self.status["GOOD"]:                       # wenn kein Tile geholt werden konnte
      if status==self.status["INVALID"]:                  # wenn es ein illegales Tile ist
        return(None, False)                               # Tile existiert nicht - muss also nicht erneut geladen werden
      return(None, True)                                  # Tile ist [noch] nicht verfügbar, refresh anfordern
    if im is None:
      im=Image.new("RGB", self.__getBitmapSize(), (204, 204, 204))
    if hndoi:                                             # wenn Merker für "nackiges Tile ausliefern" gesetzt ist
      return(im, False)                                   # Tile liegt vor und enthält bekanntermaßen keine Tracks/Points
    draw=ImageDraw.Draw(im)                               # Tile in den "Edit-Modus" laden
    if self.show_grid:
      draw.line((0, self.ts-1, 0, 0, self.ts-1, 0), fill=(0, 0, 0), width=1)
      draw.text((10, 10), "%d %d"%(x, y), (0, 0, 0))
    if self.__drawOnTile(draw, x, y, z, t):               # Tracks/Points auf das Tile zeichnen
      t.img=im                                            # Tile speichern
    return(im, False)                                     # Tile samt "liegt vor"-Kennung zurückmelden


  # ######################################################################
  # Schreibt Track-Abschnitte und Punkte auf das Image "draw", sofern
  # in self.all_tracks oder self.all_points entsprechende Daten für das
  # Tile (x, y) abgelegt sind. In diesem Fall wird True zurückgeliefert.
  # Waren keine Daten zu (x, y) abgelegt (wurde "draw" also nicht
  # verändert), wird False geliefert.
  def __drawOnTile(self, draw, x, y, z, t):
    was_modified=False
    for handle, data in self.all_tracks.items():  # über alle Track-Handle
      if handle in t.no_tracks:                   # wenn der Track nicht auf dem Tile liegt
        continue                                  # diesen Track überspringen
      if (x, y) in data.dictTN:                   # wenn Track-Daten zum Tile (im aktuellen Zoom-Level) existieren
        lst=data.dictTN[(x, y)]                   # Teil-Tracks laden (ein Track kann mehrfach über ein Tile verlaufen)
        for l in lst:                             # über alle Teil-Tracks
          draw.line(l, fill=self.all_tracks[handle].color, width=self.all_tracks[handle].width) # Teil-Track zeichnen
        t.has_tracks.add(handle)                  # und zum Tile vermerken, dass dieser Track draufgezeichnet wurde
        was_modified=True                         # Kennung für "Tile wurde verändert" setzen
      else:                                       # aktueller Track liegt nicht auf diesem Tile
        t.no_tracks.add(handle)                   # zum Tile vermerken, dass dieser Track nicht darauf liegt
    for handle, data in sorted(self.all_points.items()):  # und analog über alle Punkt-Listen-Handle
      nodupe=dict()   # zum Ausblenden von übereinander liegenden Points
      # der nodupe wird innerhalb der Schleife neu initialisiert, damit die neusten
      # Points über den älteren Points (bzgl. handle) gezeichnet werden
      if handle in t.no_points:                   # wenn diese Point-Liste nicht auf dem Tile liegt
        continue                                  # Point-Liste überspringen
      if (x, y) in data.dictTN:                   # wenn Points-Daten zum Tile (im aktuellen Zoom-Level) existieren
        lst=data.dictTN[(x, y)]                   # Points laden, die auf diesem Tile liegen
        for p, r, c, d in lst:                    # über alle Points auf diesem Tile
          p2=(p[0]/4, p[1]/4)                     # für Point-Ausblendung die Auflösung verringern
          if p2 not in nodupe:                    # wenn an p2 noch kein Point gezeichnet wurde
            nodupe.update({p2:None})              # p2 als "bereits gezeichnet" vermerken
            draw.ellipse((p[0]-r, p[1]-r, p[0]+r, p[1]+r), c, (0, 0, 0))  # Point zeichnen
#            draw.ellipse((p[0]-r, p[1]-r, p[0]+r, p[1]+r), c, (255, 0, 0))  # Point zeichnen
        t.has_points.add(handle)                  # und zum Tile vermerken, dass dieser Point draufgezeichnet wurde
        was_modified=True                         # Kennung für "Tile wurde verändert" setzen
      else:                                       # aktuelle Point-Liste liegt nicht auf diesem Tile
        t.no_points.add(handle)                   # zum Tile vermerken, dass diese Point-Liste nicht darauf liegt
    return(was_modified)


  # ######################################################################
  # Liefert die Anzahl der noch unbeantworteten Tile-Requests.
  def getOpenRequests(self):
    return(self.tc.getQueueLength())


  # ######################################################################
  # Stellt bei Übergabe von True ein, dass Linien an Tile-Grenzen
  # dargestellt werden.
  def setShowGrid(self, show_grid):
    self.show_grid=show_grid


  # ######################################################################
  # Erzwingt einen Neuaufbau des Bildes beim nächsten drawMap().
  # Ist immer dann aufzurufen, wenn sich der Inhalt des Bitmaps geändert
  # haben sollte und dies nicht auf Fenster-/Randgröße, Zoom-Level
  # oder fehlende Tiles zurückzuführen ist. Also etwa bei Änderung von
  # Track- oder Point-Listen.
  def forceRefresh(self, handle=0):
    self.v[handle].needs_refresh=True


  # ######################################################################
  # Lon./lat. to tile numbers
  # Quelle: http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Python
  def deg2num(self, latlon, zoom):
    x, y=self.deg2numF(latlon, zoom)
    return(int(x), int(y))
  def deg2numF(self, (lat_deg, lon_deg), zoom):
    try:
      lat_rad = math.radians(lat_deg)
      n = 2.0 ** zoom
      xtile = (lon_deg + 180.0) / 360.0 * n
      ytile = (1.0 - math.log(math.tan(lat_rad) + (1 / math.cos(lat_rad))) / math.pi) / 2.0 * n
    except:
      print "error deg2num()"
      return(0, 0)
    return(xtile, ytile)


  # ######################################################################
  # Tile numbers to lon./lat.
  # Quelle: http://wiki.openstreetmap.org/wiki/Slippy_map_tilenames#Python
  def num2deg(self, (xtile, ytile), zoom):
    n = 2.0 ** zoom
    lon_deg = xtile / n * 360.0 - 180.0
    lat_rad = math.atan(math.sinh(math.pi * (1 - 2 * ytile / n)))
    lat_deg = math.degrees(lat_rad)
    return(lat_deg, lon_deg)


  # ######################################################################
  # Liefert für zwei Längen/Breitengrade deren Abstand in Pixel.
  def distanceLatLonToPixel(self, pos1LL, pos2LL):
    t1TN=self.deg2numF(pos1LL, self.zoom)                                 # lat/lon nach TileNummer+Nachkommastellen
    t2TN=self.deg2numF(pos2LL, self.zoom)
    tdist=(t2TN[0]-t1TN[0], t2TN[1]-t1TN[1])                              # Abstand in Tiles
    distXY=(int(round(tdist[0]*self.ts)), int(round(tdist[1]*self.ts)))   # Abstand nach Pixel umrechnen
    return(distXY)


  # ######################################################################
  # Liefert zu einem LatLon-Wert "posLL" den um "deltaXY" Pixel
  # verschobenen LatLon-Wert.
  def getLatLonPlusPixel(self, posLL, deltaXY, zoom=None):
    if zoom is None:  zoom=self.zoom
    xfTN, yfTN=self.deg2numF(posLL, zoom)
    xiTN, yiTN=int(xfTN), int(yfTN)                           # das Tile, auf dem posLL liegt
    xb, yb=int((xfTN-xiTN)*self.ts), int((yfTN-yiTN)*self.ts) # die Position auf dem Tile in Pixel
    xn, yn=xb+deltaXY[0], yb+deltaXY[1]                       # die neue Position in Pixel relativ zu posLL
    tdx, tdy=xn/self.ts, yn/self.ts                           # die Anzahl der Tiles zwischen den Positionen
    nx, ny=xn-tdx*self.ts, yn-tdy*self.ts                     # die Position in Pixel auf dem Ziel-Tile
    rx, ry=float(nx)/self.ts, float(ny)/self.ts               # die Nachkommastellen der Zielposition
    tpx, tpy=xiTN+tdx+rx, yiTN+tdy+ry                         # die Zielkoordinaten des Tiles
    return(self.num2deg((tpx, tpy), zoom))


  # ######################################################################
  # Lädt eine Liste mit Track-Punkten.
  # Erwartetes Eingangs-Format:   [((lat, lon), data), ...]
  # Zurückgegeben wird ein Handle, mit dem dieser Track wieder
  # gelöscht werden kann.
  def loadTrack(self, trackLL, color=None, width=2):
    if len(self.all_tracks)>0:                                  # neues Handle bestimmen
      handle=max(self.all_tracks)+1
    else:
      handle=1
    track=OSMtrack()                                            # Speicher anlegen
    track.listLL=trackLL                                        # Track in lat/lon darin speichern
    track.dictTN=self.__buildDataForGPXTrack(trackLL)           # Track im aktuellen Zoom-Level nach x/y-Koordinaten wandeln
    track.dictTN_z.update({self.zoom:copy.copy(track.dictTN)})  # Track in x/y-Koordinaten zum Zoom-Level cachen
    track.valid_for_zoom=self.zoom                              # Zoom-Level für dictTN merken
    if color is not None:
      track.color=color                                         # Track-Farbe speichern
    track.width=width                                           # Track-Stärke speichern
    self.all_tracks.update({handle:track})                      # Track zum Handle global speichern
    return(handle)                                              # Track-Handle zurückliefern


  # ######################################################################
  # Setzt Farbe und Stärke, in der der Track "handle" gezeichnet wird.
  def setTrackPen(self, handle, color=None, width=2):
    if handle in self.all_tracks:                               # wenn "handle" als Track existiert
      if color is None:                                         # wenn Farbe nicht angegeben wurde
        self.all_tracks[handle].color=(199, 21, 133)            # Default-Track-Farbe einstellen
      else:                                                     # wenn Farbe angegeben wurde
        self.all_tracks[handle].color=color                     # angegebene Track-Farbe einstellen
      self.all_tracks[handle].width=width                       # Track-Stärke einstellen
      del_lst=list()
      for xyzTN, t in self.tiles.items():                       # alle Tiles löschen, auf denen...
        if handle in t.has_tracks:                              # ...dieser Track (in voriger Farbe/Stärke)...
          del_lst.append(xyzTN)                                 # ...liegt.
      for xyzTN in del_lst:
        del self.tiles[xyzTN]


  # ######################################################################
  # Entfernt den Track zum Kenner "handle" wieder.
  def unloadTrack(self, handle):
    if handle is not None:
      del_lst=list()
      for xyzTN, t in self.tiles.items():     # über alle Tiles, auf denen Tracks/Points gezeichnet wurden
        if handle in t.has_tracks:            # wenn der Track auf dem Tile liegt
          del_lst.append(xyzTN)               # Tile zum Löschen vormerken
        elif handle in t.no_tracks:           # wenn der Track nicht auf dem Tile liegt
          t.no_tracks.remove(handle)          # diese Markierung löschen (weil es diesen Track ja nicht mehr gibt)
      for xyzTN in del_lst:                   # über alle Löschvormerkungen
        del self.tiles[xyzTN]                 # Tile samt Verwaltungsdaten löschen
      try:
        del self.all_tracks[handle]           # Track löschen
      except:
        pass


  # ######################################################################
  # Erstellt aus einem Track im Format [((lat, lon), data), ...] ein
  # Dictionary mit
  #   key=(tilenummerX, tilenummerY) und
  #   value=[[x1, y1, x2, y2, x3, y3, ...], ...]
  # Wobei "value" eine Liste von Linien-Zügen darstellt.
  # Erster und letzter Wert von x/y liegen üblicherweise außerhalb des
  # Tiles. Es gilt:
  # Wird zwischen zwei Punkten das Tile verlassen, werden beide Punkte zu
  # beiden Tiles abgelegt (damit durchgehende Tracks entstehen).
  # Wird ein Tile erneut betreten, wird eine neue Punkte-Liste zum
  # Tile hinzugefügt.
  # Werden komplette Tiles übersprungen, werden alle die Tiles
  # aufgenommen, von denen mindestens eine ihrer vier Seiten einen
  # Schnittpunkt mit dem Track besitzt.
  # Bei Linien-Stärken größer Eins kann es dazu kommen, dass die Linien-
  # Stärke an Tile-Übergängen zu dünn angezeigt wird.
  def __buildDataForGPXTrack(self, trackLL):
    out=dict()
    if len(trackLL)==0:
      return(out)
    latlon, data=trackLL[0]                                                 # Init der Tile-Änderungserkennung
    x, y=self.deg2numF(latlon, self.zoom)
    xyTN_prev=(int(x), int(y))                                              # Vorgänger-Tile
    xym_prev=(int((x-xyTN_prev[0])*self.ts), int((y-xyTN_prev[1])*self.ts)) # Vorgänger-Punkt auf Tile
    lst=list()                                                              # sammelt die Track-Punkte auf dem aktuellen Tile
    for idx in xrange(len(trackLL)):
      latlon, data=trackLL[idx]                               # Data abtrennen
      x, y=self.deg2numF(latlon, self.zoom)                   # Tile-Nummer in float
      xyTN=(int(x), int(y))                                   # Tile-Nummer (in int)
      xym=(int((x-xyTN[0])*self.ts), int((y-xyTN[1])*self.ts))  # x/y auf dem Tile (Nachkommastellen*Tilesize)
      if xyTN==xyTN_prev:                                     # selbes Tile wie zuvor
        lst.extend(xym)                                       # Punkt(neu) zufügen
        xyTN_prev, xym_prev=xyTN, xym                         # Vorgänger-Tile und -Punkt neu setzen
      else:                                                   # anderes Tile als zuvor
        x, y=self.__calcDeltaOnTileBorders(xyTN, xyTN_prev)   # Korrekturwerte für Tile-Wechsel holen
        lst.extend((xym[0]+x, xym[1]+y))                      # Punkt(neu) zu Tile(alt) zufügen
        self.__appendElementToDictValue(out, xyTN_prev, lst)  # und in Ergebnis-Dict ablegen
        if abs(x)>=self.ts or abs(y)>=self.ts:                # wenn komplette Tiles übersprungen wurden
          if x==0:                                                                # wenn Tiles nur in der Senkrechten übersprungen wurden
            rtg=y/abs(y)                          # Richtung: -1 oder 1
            if y/self.ts>1000:  return(None)      # Abbruch bei irrem Track
            for y2 in range(rtg, y/self.ts, rtg):
              yo=y2*self.ts
              yn=y-y2*self.ts
              self.__appendElementToDictValue(out, (xyTN_prev[0], xyTN_prev[1]+y2), [xym_prev[0], xym_prev[1]-yo, xym[0], xym[1]+yn])
          elif y==0:                                                              # wenn Tiles nur in der Waagerechten übersprungen wurden
            rtg=x/abs(x)                          # Richtung: -1 oder 1
            if x/self.ts>1000:  return(None)      # Abbruch bei irrem Track
            for x2 in range(rtg, x/self.ts, rtg):
              xo=x2*self.ts
              xn=x-x2*self.ts
              self.__appendElementToDictValue(out, (xyTN_prev[0]+x2, xyTN_prev[1]), [xym_prev[0]-xo, xym_prev[1], xym[0]+xn, xym[1]])
          else:                                                                   # Tiles wurden auf beiden Achsen übersprungen
            rtg=x/abs(x), y/abs(y)
            xd, yd=xyTN[0]-xyTN_prev[0], xyTN[1]-xyTN_prev[1]                     # Anzahl Tiles zwischen den Punkten
            xn, yn=xym[0]-xym_prev[0]+xd*self.ts, xym[1]-xym_prev[1]+yd*self.ts   # Pixel-Delta zwischen den Punkten
            if yd+rtg[1]>1000 or xd+rtg[0]>1000:  return(None)                    # Abbruch bei irrem Track
            for y2 in range(0, yd+rtg[1], rtg[1]):
              for x2 in range(0, xd+rtg[0], rtg[0]):
                if (x2==0 and y2==0) or (x2==xd and y2==yd):                      # Start- und Ziel-Tile können ignoriert werden
                  continue
                xo, yo=x2*self.ts, y2*self.ts                                     # Pixel-Delta vom aktuellen Tile zum Start-Tile
                xn, yn=x-x2*self.ts, y-y2*self.ts                                 # Pixel-Delta vom aktuellen Tile zum Ziel-Tile
                if self.__lineSquareIntersect((xym_prev[0]-xo, xym_prev[1]-yo), (xym[0]+xn, xym[1]+yn), self.ts):
                  self.__appendElementToDictValue(out, (xyTN_prev[0]+x2, xyTN_prev[1]+y2), [xym_prev[0]-xo, xym_prev[1]-yo, xym[0]+xn, xym[1]+yn])
        lst=list()                                            # frische Liste für Tile(neu) anlegen
        lst.extend((xym_prev[0]-x, xym_prev[1]-y))            # Punkt(alt)
        lst.extend(xym)                                       # und Punkt(neu) zufügen
        xyTN_prev, xym_prev=xyTN, xym                         # Vorgänger-Tile und -Punkt neu setzen
    self.__appendElementToDictValue(out, xyTN_prev, lst)      # letzte Liste auch noch ins Ergebnis-Dict
    return(out)


  # ######################################################################
  # Liefert True, wenn die Linie (p0, p1) das Quadrat an (0, 0) mit den
  # Abmessungen (wh, wh) schneidet.
  def __lineSquareIntersect(self, (p0_x, p0_y), (p1_x, p1_y), wh):
    sl=[  ((0, 0),  (0, wh)),   # obene Seite
          ((0, 0),  (wh, 0)),   # linke Seite
          ((0, wh), (wh, wh)),  # untere Seite
          ((wh, 0), (wh, wh)) ] # rechte Seite
    for ((p2x, p2y), (p3x, p3y)) in sl:
      if self.lineIntersection((p0_x, p0_y), (p1_x, p1_y), (p2x, p2y), (p3x, p3y)):
        return(True)
    return(False)


  # ######################################################################
  # Liefert True, wenn sich die Linien (A, B) und (C, D) schneiden.
  def lineIntersection(self, A, B, C, D):
    return(self.lineIntersectionPoint(A, B, C, D) is not None)


  # ######################################################################
  # Liefert den Schnittpunkt der Linien (A, B) mit (C, D) oder None, wenn
  # sich die Linien nicht schneiden.
  # https://stackoverflow.com/a/1968345/3588613 (in den Kommentaren)
  # ...mit einem zusätzlichen float(), damit er auch bei Übergabe von
  # Integers noch korrekte Ergebnisse liefert.
  def lineIntersectionPoint(self, A, B, C, D): 
    Bx_Ax = B[0] - A[0]
    By_Ay = B[1] - A[1] 
    Dx_Cx = D[0] - C[0] 
    Dy_Cy = D[1] - C[1] 
    determinant = float(-Dx_Cx * By_Ay + Bx_Ax * Dy_Cy) 
    if abs(determinant) < 1e-20: 
      return None
    s = (-By_Ay * (A[0] - C[0]) + Bx_Ax * (A[1] - C[1])) / determinant 
    t = ( Dx_Cx * (A[1] - C[1]) - Dy_Cy * (A[0] - C[0])) / determinant 
    if s >= 0 and s <= 1 and t >= 0 and t <= 1: 
      return (A[0] + (t * Bx_Ax), A[1] + (t * By_Ay)) 
    return None


  # ######################################################################
  # Liefert die zu addierenden Werte für x und y, um die Koordinaten aus
  # einem anderen Tile verwenden zu können.
  # Es wird der Wert für den AUSTRITT aus einem Tile geliefert.
  # Für den EINTRITT in ein Tile sind x und y zu negieren.
  def __calcDeltaOnTileBorders(self, xyi_cur, xyi_old):
    xd=(xyi_cur[0]-xyi_old[0])*self.ts
    yd=(xyi_cur[1]-xyi_old[1])*self.ts
    return(xd, yd)


  # ######################################################################
  # Hängt das Element "elem" der value-Liste des Dictionaries "dic" beim
  # key "key" an.
  def __appendElementToDictValue(self, dic, key, elem):
    if key in dic:
      l=dic[key]
      l.append(elem)
      dic.update({key:l})
    else:
      dic.update({key:[elem]})
    return(dic)


  # ######################################################################
  # Liefert die Daten zum Trackpoint "index" des Tracks "handle".
  def getTrackPointData(self, handle, index):
    if handle in self.all_tracks:
      data=self.all_tracks[handle].listLL[index][1]
      return(data)
    return(None)


  # ######################################################################
  # Wie __getNewMapPointsForTrack(), nur mit vorgeschaltetem Cache pro
  # Zoom-Level.
  def getNewMapPointsForTrack(self, handle, zoom=None):
    if zoom is None:  zoom=self.zoom
    if handle in self.all_tracks:
      if zoom in self.all_tracks[handle].newMapPoints:
        return(self.all_tracks[handle].newMapPoints[zoom])
      else:
        nmp=self.__getNewMapPointsForTrack(handle, zoom)
        self.all_tracks[handle].newMapPoints.update({zoom:nmp})
        return(nmp)
    return({})


  # ######################################################################
  # Liefert für den Track "handle" ein Dictionary mit NewMapPoints.
  # Der Key ist jeweils der Index im Track, der vor dem NewMapPoint liegt.
  # Der Value ist eine Liste aus Tupeln (nummer, (lat, lon)). Diese Liste
  # enthält dann mehr als ein Tupel, wenn mehrere NewMapPoints zwischen
  # zwei aufeinander folgenden TrackPoints liegen. "nummer" entspricht dem
  # Index in der Liste.
  # Das Rückgabe-Dictionary enthält somit die Koordinaten, an denen beim
  # Nachfahren eines Tracks neue Bitmaps erstellt werden sollten bzw.
  # doMoveMap() nicht mehr sicher innerhalb des "Borders" liegt.
  #
  # Beispiel zum Inhalt der Rückgabe:
  #   { 26 : [(1, (54.xxx, 9.xxx))],
  #     33 : [(1, (54.xxx, 9.xxx)), (2, (54.xxx, 9.xxx))],
  #     41 : [(1, (54.xxx, 9.xxx))] }
  def __getNewMapPointsForTrack(self, handle, zoom):
    ret_dict=dict()
    trk=self.all_tracks[handle].listLL
    baseLL=trk[0][0]                                # Position des ersten bzw. vorigen NewMapPoints in Geo-Koordinaten
    base_xTN, base_yTN=self.deg2numF(baseLL, zoom)  # Position des ersten bzw. vorigen NewMapPoints in Tiles
    prev_delta_x, prev_delta_y=0, 0                 # voriger Abstand zu baseLL in Pixel
    #self.__appendElementToDictValue(ret_dict, 0, (1, baseLL)) # ersten Trackpunkt unbedingt einfügen
    idx=1
    while idx<len(trk):
      cur_xTN, cur_yTN=self.deg2numF(trk[idx][0], zoom)                                                         # Position des aktuellen TrackPoints in Tiles
      cur_delta_x, cur_delta_y=(int(round((cur_xTN-base_xTN)*self.ts)), int(round((cur_yTN-base_yTN)*self.ts))) # dessen aktueller Abstand zu baseLL in Pixel
      if abs(cur_delta_x)>self.ts or abs(cur_delta_y)>self.ts:
        # wenn der aktuelle Trackpunkt schon über "tilesize" Pixel von "base" entfernt ist, ...
        # ...muss der gesuchte Punkt zwischen dem vorigen und dem aktuellen Trackpunkt liegen
        lnXY=self.getLine((prev_delta_x, prev_delta_y), (cur_delta_x, cur_delta_y))       # Linie zwischen vorigen und aktuellen Trackpunkt legen
        # die Tupel in lnXY enthalten den Abstand zu baseLL
        tc=1  # die Linie kann über mehrere Tiles verlaufen
        for xl, yl in lnXY:
          if abs(xl)>tc*self.ts or abs(yl)>tc*self.ts:
            tpx, tpy=xl, yl                                                               # der Abstand des gesuchten Punktes zum vorigen Trackpunkt
            base2LL=self.getLatLonPlusPixel(baseLL, (xl, yl), zoom)
            self.__appendElementToDictValue(ret_dict, idx-1, (tc, base2LL))
            tc+=1
        baseLL=base2LL
        base_xTN, base_yTN=self.deg2numF(baseLL, zoom)
        prev_delta_x, prev_delta_y=cur_delta_x-tpx, cur_delta_y-tpy
      else:
        prev_delta_x, prev_delta_y=cur_delta_x, cur_delta_y   # aktuellen Abstand zu baseLL als vorigen Abstand zu baseLL setzen
      idx+=1
    self.__appendElementToDictValue(ret_dict, idx-1, (1, trk[-1][0])) # letzten Trackpunkt unbedingt einfügen
    return(ret_dict)


  # ######################################################################
  # Liefert eine Delta-Strecke in Pixeln, die bei Track[index] beginnt und
  # bei NewMapPoint[index] endet.
  # Der Punkt NewMapPoint[index] ist immer >=Track[index] - bezogen auf
  # den Track.
  # Immer dann, wenn zwischen zwei TrackPoints mehrere Tiles liegen, sind
  # zum NewMapPoint[index] mehrere Punkte gespeichert. Zur Auswahl eines
  # solchen Punktes dient "sub_index".
  # Für den Startpunkt gilt:
  #   sub_index==0  - index kennzeichnet einen Track[index]
  #   sub_index>0   - index kennzeichnet einen NewMapPoint[index]
  #
  # Rückgabe ist z.B.:
  # (171, 1), (54.xxx, 9.xxx), [((-1, -1), 150), ((-2, -2), 150), ... ((258, 211), 170), ((259, 212), 171)]
  # Das erste Element ist der Folge-Index im Format (index, sub_index).
  # Das zweite Element ist die Ziel-Koordinate (für die das nächste Bitmap asynchron vorberechnet werden muss).
  # Das dritte Element ist eine Liste von Tupeln im Format ((delta_x, delta_y), index).
  def getTrackPartAsPixelDelta(self, handle, index, sub_index=0):
    if handle not in self.all_tracks: return(None, (0, 0), [])
    trk=self.all_tracks[handle].listLL
    if index>=len(trk)-1:  return(None, (0, 0), [])
    newMapPoints=self.getNewMapPointsForTrack(handle)
    prev_mp_idx, next_mp_idx=self.__findNearestDictKeys(newMapPoints, index+1) # Nachfolge-Index

    if prev_mp_idx==index:  # Trackpunkt liegt unmittelbar vor NewMapPoint
      cur_mp_idx=prev_mp_idx
    else:
      cur_mp_idx=next_mp_idx

    # Index vom Endpunkt bestimmen (ist Teil der Rückgabe)
    next_idx=None
    if sub_index==0:
      next_idx=(cur_mp_idx, 1)
    elif sub_index>0:
      pt_cnt=len(newMapPoints[cur_mp_idx])
      if pt_cnt>1:                      # mehrere Punkte hinter newMapPoints[next_mp_idx]
        if sub_index<pt_cnt:
          next_idx=(prev_mp_idx, sub_index+1)
        elif pt_cnt==sub_index:
          next_idx=(next_mp_idx, 1)
      else: # kann nur pt_cnt==1 sein
        next_idx=(next_mp_idx, 1)

    # Liste aller zu passierenden Punkte erstellen
    lstLL=list()  # Format: [(index, sub_index), (index, sub_index), ...]
    if sub_index==0:                            # Start auf TrackPoint
      for idx in range(index, next_idx[0]+1):
        lstLL.append((idx, 0))
      lstLL.append((next_idx[0], 1))
    elif sub_index>0:                           # Start auf NewMapPoint
      if index==next_idx[0] and next_idx[1]>1:  # wenn direkt ein NewMapPoint folgt
        lstLL.append((index, sub_index))
        lstLL.append((index, sub_index+1))
      else:                                     # wenn ein TrackPoint folgt
        lstLL.append((index, sub_index))
        for idx in range(index+1, next_idx[0]+1):
          lstLL.append((idx, 0))
        lstLL.append((next_idx[0], 1))

    # Linien zwischen die zu passierenden Punkte legen
    lstXY=list()
    base=(0, 0)
    for idx in range(len(lstLL)):
      if lstLL[idx][1]==0:  p=trk[lstLL[idx][0]][0]
      else:                 p=newMapPoints[lstLL[idx][0]][lstLL[idx][1]-1][1]
      if idx==0:
        p1=p
      else:
        p2=p
        dxy=self.distanceLatLonToPixel(p1, p2)
        lstXY.append((lstLL[idx-1][0], self.getLine(base, dxy)))
        base=dxy

    # Linien-Listen zu einer Gesamt-Liste verschmelzen
    single_list=list()
    for lst_idx in range(len(lstXY)):
      idx=lstXY[lst_idx][0]
      for xd, yd in lstXY[lst_idx][1][1:]:
        single_list.append(((xd, yd), idx))
    return(next_idx, newMapPoints[next_idx[0]][next_idx[1]-1][1], single_list)


  # ######################################################################
  # Liefert vom Dictionary "dic" die beiden Keys, deren Differenz zu "key"
  # am kleinsten ist. Existiert "key" als Key in "dic", wird er mit seinem
  # Vorgänger geliefert.
  # Enthält "dic" keinen Key >="key", wird None geliefert.
  # Beispiel:
  #   d= {10:1, 20:2, 30:3, 40:4, 50:5}
  #   (d,  5) -> (None, 10)
  #   (d, 10) -> (None, 10)
  #   (d, 25) -> (20, 30)
  #   (d, 30) -> (20, 30)
  #   (d, 50) -> (40, 50)
  #   (d, 51) -> (None, None)
  def __findNearestDictKeys(self, dic, key):
    p=None
    for k in sorted(dic):
      if k>=key:
        return((p, k))
      p=k
    return((None, None))


  # ###########################################################
  # Get all points of a straight line.
  # Quelle: http://stackoverflow.com/a/25913345/3588613
  def getLine(self, (x1, y1), (x2, y2)):
    points = []
    issteep = abs(y2-y1) > abs(x2-x1)
    if issteep:
        x1, y1 = y1, x1
        x2, y2 = y2, x2
    rev = False
    if x1 > x2:
        x1, x2 = x2, x1
        y1, y2 = y2, y1
        rev = True
    deltax = x2 - x1
    deltay = abs(y2-y1)
    error = int(deltax / 2)
    y = y1
    ystep = None
    if y1 < y2:
        ystep = 1
    else:
        ystep = -1
    for x in range(x1, x2 + 1):
        if issteep:
            points.append((y, x))
        else:
            points.append((x, y))
        error -= deltay
        if error < 0:
            y += ystep
            error += deltax
    # Reverse the list if the coordinates were reversed
    if rev:
        points.reverse()
    return points


  # ######################################################################
  # Quelle: https://pypi.python.org/pypi/haversine
  # AVG_EARTH_RADIUS = 6371  - in km
  def haversine(self, point1, point2, miles=False):
    # unpack latitude/longitude
    lat1, lng1 = point1
    lat2, lng2 = point2

    # convert all latitudes/longitudes from decimal degrees to radians
    lat1, lng1, lat2, lng2 = map(math.radians, (lat1, lng1, lat2, lng2))

    # calculate haversine
    lat = lat2 - lat1
    lng = lng2 - lng1
    d = math.sin(lat * 0.5) ** 2 + math.cos(lat1) * math.cos(lat2) * math.sin(lng * 0.5) ** 2
    h = 2 * 6371 * math.asin(math.sqrt(d))
    if miles:
        return h * 0.621371  # in miles
    else:
        return h  # in kilometers


  # ######################################################################
  # Lädt eine Liste von Punkten samt Zusatzdaten.
  # Erwartetes Eingangs-Format:
  #   [((lat, lon), radius, farbe, data), ((lat, lon), radius,  farbe, data), ...]
  # Zurückgegeben wird ein Handle, mit dem diese Punktliste wieder
  # gelöscht werden kann.
  def loadPoints(self, pointsLL):
    if len(self.all_points)>0:
      handle=max(self.all_points)+1
    else:
      handle=1
    points=OSMpoints()
    points.listLL=pointsLL
    points.dictTN=self.__buildDataForGPXPoints(pointsLL)
    points.dictTN_z.update({self.zoom:copy.copy(points.dictTN)})
    points.valid_for_zoom=self.zoom
    self.all_points.update({handle:points})
    return(handle)


  # ######################################################################
  # Entfernt die Punkt-Liste zum Kenner "handle" wieder.
  def unloadPoints(self, handle):
    if handle is not None:
      del_lst=list()
      for xyzTN, t in self.tiles.items():     # über alle Tiles
        if handle in t.has_points:            # wenn Punkte der zu löschenden Punkte-Liste drauf sind
          del_lst.append(xyzTN)               # Tile zum Löschen vormerken
        elif handle in t.no_points:           # wenn die Punkte-Liste als nicht drauf liegend vermerkt war
          t.no_points.remove(handle)          # weg mit diesem Handle
      for xyzTN in del_lst:                   # über alle Lösch-Merker
        del self.tiles[xyzTN]                 # Löschung ausführen
      try:
        del self.all_points[handle]           # und noch die Punkte-Liste selbst killen
      except:
        pass


  # ######################################################################
  # Erstellt aus einer Punkt-Liste im Format
  #     [((lat, lon), radius, color, data), ...]
  # ein Dictionary mit
  #   key=(tilenummerX, tilenummerY) und
  #   value=[((x, y), radius, color), ...]
  # Ein Punkt mit einem "radius">1 kann bis zu vier Tiles überdecken.
  # "radius" muss kleiner Tilesize sein!
  def __buildDataForGPXPoints(self, pointsLL):
    out=dict()
    for idx in xrange(len(pointsLL)):
      latlon, radius, col, data=pointsLL[idx]
      x, y=self.deg2numF(latlon, self.zoom)
      xyTN=(int(x), int(y))
      xym=(int(round((x-xyTN[0])*self.ts)), int(round((y-xyTN[1])*self.ts)))
      xyTNadd=list()
      if xym[0]-radius<0:                                                 # Punkt ragt ins linke Tile herein
        xyTNadd.append(((xyTN[0]-1, xyTN[1]), (xym[0]+self.ts, xym[1])))
      elif xym[0]+radius>=self.ts:                                        # Punkt ragt ins rechte Tile herein
        xyTNadd.append(((xyTN[0]+1, xyTN[1]), (xym[0]-self.ts, xym[1])))
      if xym[1]-radius<0:                                                 # Punkt ragt ins darüber-liegende Tile herein
        xyTNadd.append(((xyTN[0], xyTN[1]-1), (xym[0], xym[1]+self.ts)))
      elif xym[1]+radius>=self.ts:                                        # Punkt ragt ins darunter-liegende Tile herein
        xyTNadd.append(((xyTN[0], xyTN[1]+1), (xym[0], xym[1]-self.ts)))
      if len(xyTNadd)>=2:                                                 # wenn er horizontal und vertikal überlappt, gibts noch ein drittes Tile
        tmp=((xyTNadd[0][0][0], xyTNadd[1][0][1]), (xyTNadd[0][1][0], xyTNadd[1][1][1]))
        xyTNadd.append(tmp)
      self.__appendElementToDictValue(out, xyTN, (xym, radius, col, data))
      for tn, xy in xyTNadd:
        self.__appendElementToDictValue(out, tn, (xy, radius, col, data))
    return(out)


  # ######################################################################
  # Liefert zur Position "posXY" den Inhalt von "data" der darunter
  # liegenden Points samt ihrer aktuellen XY-Position als Liste von Tupeln
  # oder [], wenn kein Punkt darunter liegt.
  def getDataForPoint(self, posXY, handle=0):
    tile0iLL=self.num2deg(self.v[handle].tile0TN, self.zoom)                          # lat/lon des Eckpunktes vom Tile auf (0, 0)
    tile0d0XY=self.distanceLatLonToPixel(self.v[handle].pos0LL, tile0iLL)             # Abstand Eckpunkt zu (0, 0)
    distXY=(posXY[0]-tile0d0XY[0], posXY[1]-tile0d0XY[1])                             # Abstand Eckunkt vom Tile auf (0, 0) zu posXY
    tdist=(int(distXY[0]/self.ts), int(distXY[1]/self.ts))                            # Abstand von posXY zu (0, 0) in Tiles
    tilecTN=(self.v[handle].tile0TN[0]+tdist[0], self.v[handle].tile0TN[1]+tdist[1])  # das Tile, das unter posXY liegt
    disttXY=(distXY[0]-tdist[0]*self.ts, distXY[1]-tdist[1]*self.ts)                  # Abstand von posXY zum Eckpunkt vom Tile unter posXY
    #print tile0d0XY, self.v[handle].tile0TN, distXY, tdist, tilecTN, disttXY
    dlst=list()
    for h in self.all_points:                                                   # über alle Punkt-Listen
      if tilecTN in self.all_points[h].dictTN:                                  # wenn Punkt-Liste überhaupt das Tile zu posXY enthält
        for xy, r, c, data in self.all_points[h].dictTN[tilecTN]:               # über alle Punkte auf diesem Tile
          if xy[0]-r<disttXY[0]<=xy[0]+r and xy[1]-r<disttXY[1]<=xy[1]+r:       # wenn der Punkt unter posXY liegt
            xya=(xy[0]+tdist[0]*self.ts+tile0d0XY[0], xy[1]+tdist[1]*self.ts+tile0d0XY[1])  # dessen derzeitigen Mittelpunkt bestimmen
            dlst.append((xya, data))                                            # und samt "data" der Ergebnisliste zufügen
    return(dlst)


  # ######################################################################
  # Liefert eine Liste aller Track-Handle der Tracks, die unter der
  # Fensterkoordinate "posXY" verlaufen.
  def findTrackForPoint(self, posXY, handle=0):
    tile0iLL=self.num2deg(self.v[handle].tile0TN, self.zoom)                          # lat/lon des Eckpunktes vom Tile auf (0, 0)
    tile0d0XY=self.distanceLatLonToPixel(self.v[handle].pos0LL, tile0iLL)             # Abstand Eckpunkt zu (0, 0)
    distXY=(posXY[0]-tile0d0XY[0], posXY[1]-tile0d0XY[1])                             # Abstand Eckunkt vom Tile auf (0, 0) zu posXY
    tdist=(int(distXY[0]/self.ts), int(distXY[1]/self.ts))                            # Abstand von posXY zu (0, 0) in Tiles
    tilecTN=(self.v[handle].tile0TN[0]+tdist[0], self.v[handle].tile0TN[1]+tdist[1])  # das Tile, das unter posXY liegt
    disttXY=(distXY[0]-tdist[0]*self.ts, distXY[1]-tdist[1]*self.ts)                  # Abstand von posXY zum Eckpunkt vom Tile unter posXY
    radius=3
    # Erstmal alle Tiles identifizieren, die im "radius" um "disttXY" liegen.
    # Pro Tile wird auch der relevante Suchbereich abgelegt.
    t=0
    tiles=[(self.__getInspectionRect(disttXY, radius, t), tilecTN)] # Liste aller Tiles in der Nähe von posXY
    if disttXY[0]-radius<0:
      t+=8                                                                                        # links
      tiles.append((self.__getInspectionRect(disttXY, radius, 8), (tilecTN[0]-1, tilecTN[1])))
    if disttXY[0]+radius>self.ts:
      t+=2                                                                                        # rechts
      tiles.append((self.__getInspectionRect(disttXY, radius, 2), (tilecTN[0]+1, tilecTN[1])))
    if disttXY[1]-radius<0:
      t+=1                                                                                        # oben
      tiles.append((self.__getInspectionRect(disttXY, radius, 1), (tilecTN[0], tilecTN[1]-1)))
    if disttXY[1]+radius>self.ts:
      t+=4                                                                                        # unten
      tiles.append((self.__getInspectionRect(disttXY, radius, 4), (tilecTN[0], tilecTN[1]+1)))
    if t==3:                                                                                      # rechts oben
      tiles.append((self.__getInspectionRect(disttXY, radius, t), (tilecTN[0]+1, tilecTN[1]-1)))
    elif t==6:                                                                                    # rechts unten
      tiles.append((self.__getInspectionRect(disttXY, radius, t), (tilecTN[0]+1, tilecTN[1]+1)))
    elif t==9:                                                                                    # links oben
      tiles.append((self.__getInspectionRect(disttXY, radius, t), (tilecTN[0]-1, tilecTN[1]-1)))
    elif t==12:                                                                                   # links unten
      tiles.append((self.__getInspectionRect(disttXY, radius, t), (tilecTN[0]-1, tilecTN[1]+1)))
    # Nun werden solche Bereiche gesucht, die von Tracks durchkreuzt werden
    intersect=set()
    for track, data in self.all_tracks.items():               # über alle Tracks
      for rect, tile in tiles:                                # über alle identifizierten Tiles
        p34=[ ((rect[0], rect[2]), (rect[1], rect[2])),       # InspectionRect: obere Linie
              ((rect[0], rect[3]), (rect[1], rect[3])),       # InspectionRect: untere Linie
              ((rect[0], rect[2]), (rect[0], rect[3])),       # InspectionRect: linke Linie
              ((rect[1], rect[2]), (rect[1], rect[3])) ]      # InspectionRect: rechte Linie
        if tile in data.dictTN:                               # wenn auf dem Tile ein Abschnitt des Tracks liegt
          for dictTN in data.dictTN[tile]:                    # über alle Track-Abschnitte auf diesem Tile
            for idx in range(len(dictTN)/2-1):                # über alle Linien-Abschnitte
              p1=(dictTN[idx*2],    dictTN[idx*2+1])
              p2=(dictTN[idx*2+2],  dictTN[idx*2+3])
              for p3, p4 in p34:                              # über alle vier Seiten des InspectionRect's
                if self.lineIntersection(p1, p2, p3, p4):     # wenn der Linien-Abschnitt des Tracks eine Seite des InspectionRect kreuzt
                  intersect.add(track)                        # ...liegt der Track im InspectionRect (aka Suchbereich)
                  break
              if track in intersect:  break
            if track in intersect:    break
        if track in intersect:        break
    return(list(intersect))


  # ######################################################################
  # Liefert die Eckpunkte des Rechtecks "p"+-"r" für das Tile mit dem
  # Versatz "td".
  def __getInspectionRect(self, p, r, td):
    x, y=p
    ts=self.ts
    if td==0:     return((max(0, x-r),    min(ts, x+r),     max(0, y-r),    min(ts, y+r)))      # Zentrum
    elif td==1:   return((max(0, x-r),    min(ts, x+r),     max(0, y+ts-r), min(ts, y+ts+r)))   # oben
    elif td==2:   return((max(0, x-ts-r), min(ts, x-ts+r),  max(0, y-r),    min(ts, y+r)))      # rechts
    elif td==3:   return((max(0, x-ts-r), min(ts, x-ts+r),  max(0, y+ts-r), min(ts, y+ts+r)))   # rechts oben
    elif td==4:   return((max(0, x-r),    min(ts, x+r),     max(0, y-ts-r), min(ts, y-ts+r)))   # unten
    elif td==6:   return((max(0, x-ts-r), min(ts, x-ts+r),  max(0, y-ts-r), min(ts, y-ts+r)))   # rechts unten
    elif td==8:   return((max(0, x+ts-r), min(ts, x+ts+r),  max(0, y-r),    min(ts, y+r)))      # links
    elif td==9:   return((max(0, x+ts-r), min(ts, x+ts+r),  max(0, y+ts-r), min(ts, y+ts+r)))   # links oben
    elif td==12:  return((max(0, x+ts-r), min(ts, x+ts+r),  max(0, y-ts-r), min(ts, y-ts+r)))   # links unten


  # ######################################################################
  # Liefert den Index des TrackPoints vom Track "track_handle" mit dem
  # kleinsten Abstand zur Geo-Koordinate "posLL".
  def findTrackPointIndexForPoint(self, posLL, track_handle):
    best_idx=None
    best_dist=None
    if track_handle in self.all_tracks:
      for idx in xrange(len(self.all_tracks[track_handle].listLL)):
        pt=self.all_tracks[track_handle].listLL[idx][0]
        d=self.haversine(posLL, pt)
        if best_dist is None:
          best_dist=d
          best_idx=idx
        elif best_dist>d:
          best_dist=d
          best_idx=idx
    return(best_idx)


  # ######################################################################
  # Liefert ein Tupel aus dem höchsten Zoom-Level, bei dem alle
  # Koordinaten aus "point_listLL" auf den Anzeigebereich passen und den
  # Geo-Koordinaten des Mittelpunktes.
  def getZoomAndCenterForAllPointsVisible(self, point_listLL):
    nwLL=soLL=point_listLL[0]
    for pLL in point_listLL:  # Eckpunkte suchen
      if pLL[0]>nwLL[0]:  nwLL=(pLL[0], nwLL[1])  # lat(n)
      if pLL[0]<soLL[0]:  soLL=(pLL[0], soLL[1])  # lat(s)
      if pLL[1]<nwLL[1]:  nwLL=(nwLL[0], pLL[1])  # lon(w)
      if pLL[1]>soLL[1]:  soLL=(soLL[0], pLL[1])  # lon(o)
    for zoom in range(self.max_zoom, -1, -1):
      pnwTN=self.deg2numF(nwLL, zoom) # Tile(nord westen) = links oben
      psoTN=self.deg2numF(soLL, zoom) # Tile(süd osten) = rechts unten
      dx=abs(psoTN[0]-pnwTN[0])       # Tile-Delta x
      dy=abs(pnwTN[1]-psoTN[1])       # Tile-Delta y
      if dx<self.tile_cnt_dc[0] and dy<self.tile_cnt_dc[1]:
        return(zoom, (nwLL[0]-(nwLL[0]-soLL[0])/2, soLL[1]-(soLL[1]-nwLL[1])/2))
    return(0, (0.0, 0.0))



# ######################################################################
# main()
if __name__=="__main__":
  pt=(54.805060, 9.524878)
  osm=OSMscr3((1024, 768), pt, 18, (2, 2))
  
