#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ###########################################################
# Auslagerung einiger Funktionen, die nix mit wxPython
# zu tun haben.
#
# Detlev Ahlgrimm, 2017

import time
import math
import os
import sys
from lxml import etree  # python-lxml-3.3.5-2.1.4.x86_64
from datetime import datetime, tzinfo, timedelta
from calendar import timegm
import requests

# ###########################################################
# Quelle: https://aboutsimon.com/blog/2013/06/06/Datetime-hell-Time-zone-aware-to-UNIX-timestamp.html
class UTC(tzinfo):
    """UTC"""
    def utcoffset(self, dt):
        return timedelta(0)
    def tzname(self, dt):
        return "UTC"
    def dst(self, dt):
        return timedelta(0)


class OSMhelper():
  def __init__(self):
    pass

  # ###########################################################
  # Liefert den UNIX-Timestamp für einen Timestamp-String in
  # UTC.
  # Erweitert mit: http://stackoverflow.com/a/33148723/3588613
  def getTimestampStringAsUTC(self, strg):
    if "." in strg:
      datetime_obj = datetime.strptime(strg, '%Y-%m-%dT%H:%M:%S.%fZ')
    else:
      datetime_obj = datetime.strptime(strg, '%Y-%m-%dT%H:%M:%SZ')
    datetime_obj = datetime_obj.replace(tzinfo=UTC())
    timestamp = timegm(datetime_obj.timetuple())
    return(timestamp)


  # ######################################################################
  # Liefert zu einem UNIX-Timestamp (UTC, Integer) einen String mit der
  # Ortszeit.
  def getUTCTimestampAsLocal(self, ts):
    return(datetime.fromtimestamp(ts).strftime("%Y.%m.%d %H:%M:%S"))


  # ######################################################################
  # Quelle: https://pypi.python.org/pypi/haversine
  # AVG_EARTH_RADIUS = 6371  - in km
  def haversine(self, point1, point2, miles=False):
    # unpack latitude/longitude
    lat1, lng1 = point1
    lat2, lng2 = point2

    # convert all latitudes/longitudes from decimal degrees to radians
    lat1, lng1, lat2, lng2 = map(math.radians, (lat1, lng1, lat2, lng2))

    # calculate haversine
    lat = lat2 - lat1
    lng = lng2 - lng1
    d = math.sin(lat * 0.5) ** 2 + math.cos(lat1) * math.cos(lat2) * math.sin(lng * 0.5) ** 2
    h = 2 * 6371 * math.asin(math.sqrt(d))
    if miles:
        return h * 0.621371  # in miles
    else:
        return h  # in kilometers


  # ###########################################################
  # Liefert die Rückgabe von "nominatim.openstreetmap.org" zu
  # einer Geo-Koordinate und Zoom-Level.
  def getNominatim(self, (lat, lon), zoom):
    url="http://nominatim.openstreetmap.org/reverse?format=xml&lat=%f&lon=%f&zoom=%d&addressdetails=1"%(lat, lon, zoom)
    text=""
    response=requests.get(url, timeout=1)
    try:
      root=etree.fromstring(response.content)
    except:
      print "response.content= %s\n"%(response.content,),
      return(text)
    for elem in root:
      if elem.tag=="result":
        text=elem.text.replace(", ", ", \n ")
        break
    return(text)


  # ######################################################################
  # GPX-Datei laden, parsen und als Liste von Tracks, bestehend aus Listen
  # von Tupeln ((lat, lon), timestamp), zurückgeben.
  # Bei ignore_trkseg==False werden die Tags <trk> und <trkseg>
  # gleichwertig behandelt - in beiden Fällen wird eine neue Sub-Liste
  # erstellt.
  # Bei ignore_trkseg==True wird das Tag <trkseg> ignoriert und somit der
  # aktuelle fortgesetzt.
  def loadGPX(self, filename, ignore_trkseg=True):
    try:
      with open(filename, "r") as fl:
        filedata=fl.read()
    except:
      return([], [])
    try:
      root=etree.fromstring(filedata)
    except:
      return([], [])
    ns=""
    if None in root.nsmap:
      ns="{"+root.nsmap[None]+"}"
    tracks=list()
    wpts=list()
    if etree.iselement(root) and root.tag==ns+"gpx":
      for trk in root:
        if etree.iselement(trk) and trk.tag==ns+"wpt":
          wptLL=(float(trk.get("lat")), float(trk.get("lon")))
          wptData=dict()
          for e in trk:
            if etree.iselement(e) and e.tag==ns+"name":
              wptData.update({"name":e.text})
            if etree.iselement(e) and e.tag==ns+"desc":
              wptData.update({"desc":e.text})
            if etree.iselement(e) and e.tag==ns+"type":
              wptData.update({"type":e.text})
          wpts.append((wptLL, wptData))
        if ignore_trkseg: track=list()
        if etree.iselement(trk) and trk.tag==ns+"trk":
          for trkseg in trk:
            if etree.iselement(trkseg) and trkseg.tag==ns+"trkseg":
              if not ignore_trkseg: track=list()
              for trkpt in trkseg:
                if etree.iselement(trkpt) and trkpt.tag==ns+"trkpt":
                  t=0
                  for e in trkpt:
                    if etree.iselement(e) and e.tag==ns+"time":
                      t=self.getTimestampStringAsUTC(e.text)
                      break
                  track.append(((float(trkpt.get("lat")), float(trkpt.get("lon"))), t))
              if not ignore_trkseg and len(track)>0:
                tracks.append(track)
        if ignore_trkseg and len(track)>0:
          tracks.append(track)
    if len(tracks)>0:
      self.file_loaded=True
    return(tracks, wpts)


  # ######################################################################
  # gpsxml-Datei laden, parsen und als Track im Format
  #   [((lat, lon), (timestamp, speed, dist, secs)), ...]
  # zurückgeben.
  def loadGpsXml(self, filename):
    try:
      with open(filename, "r") as fl:
        filedata=fl.read()
    except:
      return([])      
    try:
      root=etree.fromstring(filedata)
    except:
      return([])
    track=list()
    prev_posLL=None
    speed, dist, secs=(0.0, 0.0, 0)
    for point in root.iter("gps-run"):              # da sollte es nur einen von geben, aber wer weiss
      lvl1=point.getchildren()                      # eine Ebene runter
      for e in lvl1:                                # über alle gps-point's
        if e.tag.lower()=="gps-point":
          bssid=e.get("bssid")
          try:
            timestmp=int(e.get("time-sec")) #+float("0."+e.get("time-usec"))
            #if os.path.basename(gpsxml)=="Kismet-20060628-13-59-54-1.gpsxml":
            #  timestmp+=321051600
          except:
            pass
          posLL=(float(e.get("lat")), float(e.get("lon")))
          if bssid=="GP:SD:TR:AC:KL:OG":
            if prev_posLL is not None:
              dist+=self.haversine(prev_posLL, posLL)
              secs+=timestmp-pref_timestmp
              speed=round(float(e.get("spd"))*3.6, 2)
            track.append((posLL, (timestmp, speed, dist, secs)))
            prev_posLL, pref_timestmp=posLL, timestmp
    return(track)


  # ######################################################################
  # Berechnet für jeden Trackpunkt vom Track "trackLL" im Format
  #   [((lat, lon), timestamp), ...]
  # die Daten
  #   aktuelle Geschwindigkeit in km/h
  #   bisher gefahrene Kilometer
  #   bisherige Zeitdauer in Sekunden
  # und liefert den Track als Liste von Tupeln:
  #   [((lat, lon), (timestamp, speed, dist, secs)), ...]
  def calcGpxTrackpointData(self, trackLL):
    lst=list()
    if len(trackLL)==0:
      return(lst)
    prev_posLL=None
    speed, dist, secs=(0.0, 0.0, 0)
    for posLL, timestmp in trackLL:
      if prev_posLL is not None:                  # ab dem zweiten Wert...
        c_dist=self.haversine(prev_posLL, posLL)  # Entfernung zwischen n-1 und n
        c_secs=timestmp-pref_timestmp             # Dauer von n-1 bis n
        if c_secs>0:
          speed=c_dist/c_secs*3600.0
        else:   # wenn selbe Zeit wie zuvor -> Geschwindigkeit =0
          speed=0.0
        dist+=c_dist
        secs+=c_secs
      lst.append((posLL, (timestmp, speed, dist, secs)))
      prev_posLL, pref_timestmp=posLL, timestmp
    return(lst)


  # ######################################################################
  # Liefert für einen Track (als Liste aus Tupeln ((lat, lon), timestamp))
  # die Summe der Abschitts-Entfernungen in Kilometern. 
  def getTrackLength(self, track):
    dist=0
    ptl, tm=track[0]
    for pt, tm in track:
      dist+=self.haversine(ptl, pt)
      ptl=pt
    return(dist)


  # ######################################################################
  # Liefert für einen Track (als Liste aus Tupeln im Format
  #     ((lat, lon), (timestamp, speed, dist, secs, ...)) )
  # eine Liste von Punkten mit Start- und Endzeit der Koodinaten, wo
  # mindestens "min_dur_secs" lang kaum Bewegung stattfand. Wobei mit
  # "kaum Bewegung" gemeint ist, dass sämtliche Bewegung mit weniger als
  # "max_ign_spd_kmh" Stundenkilometern erfolgte und innerhalb eines
  # Umkreises von 10 Metern auftritt. Ohne die Prüfung auf den Umkreis
  # würde "langsames Fahrrad-Schieben" ggf. zu einem sehr großen Pause-
  # Gebiet führen - der zurückgelieferte Pausepunkt also deutlich abseits
  # des realen Pausepunktes liegen.
  # Die Rückgabe-Liste enthält Tupel im Format:
  #       (lat,        lon,     start-timestamp, ende-timestamp)
  # z.B.: (54.7127033, 9.76528, 1469954860,      1469955259)
  def getPausePoints(self, track, min_dur_secs=60, max_ign_spd_kmh=5):
    if len(track)==0:
      return([])
    pause_point=list()                              # Rückgabeliste initialisieren
    ptis=-1                                         # nimmt den Index von "track" eines [potentiellen] Pausen-Start-Punktes auf
    (lato, lono), tmo=track[0][0], track[0][1][0]   # Daten des Vorgängersatzes merken bzw. initialisieren
    for idx in xrange(1, len(track)):
      #if idx>=15297 and idx<19561:
      #  print "%5d - (%.6f %.6f) %d %4.1fkm/h %6.4fkm %6dsec"%(idx, track[idx][0][0], track[idx][0][1], track[idx][1][0], track[idx][1][1], track[idx][1][2], track[idx][1][3])
      (lat, lon), tm=track[idx][0], track[idx][1][0]
      if tm-tmo==0:                                 # wenn zwei Sätze die selbe Zeit haben....
        continue                                    # ...will man das nicht sehen... -> ignorieren
      spd=(track[idx-1][1][1]+track[idx][1][1])/2   # (Durchschnitt von letzter und aktueller) Geschwindigkeit in km/h
      lato, lono, tmo=lat, lon, tm
      if ptis>=0:                                   # wenn gerade eine potentielle Pause läuft
        lata=sum(latl)/float(len(latl))             # Mittelwert aller bisherigen Pause-Koordinaten bilden
        lona=sum(lonl)/float(len(lonl))
        if spd>max_ign_spd_kmh or self.haversine((lata, lona), (lat, lon))>0.01:        # ...und diese jetzt zuende wäre
          if (track[idx-1][1][0]-track[ptis][1][0])>=min_dur_secs: # ...und die Pause lang genug war...
            # ...dann ist sie nun tatsächlich zuende und liegt zwischen "ptis" und "idx-1".
            lats, lons, cnt=0, 0, 0
            #for i in range(ptis, idx):              # lat/lon vom Pausen-Punkt = Mittelwert aller Trackpoints im Zeitbereich
            #  lats+=track[i][0][0]
            #  lons+=track[i][0][1]
            #  cnt+=1
            #pause_point.append((lats/cnt, lons/cnt, track[ptis][1][0], track[idx-1][1][0]))
            pause_point.append((lata, lona, track[ptis][1][0], track[idx-1][1][0]))
          ptis=-1                                   # die Pause ist jetzt in jedem Fall zuende
        else:
          latl.append(lat)
          lonl.append(lon)
      elif ptis<0 and spd<=max_ign_spd_kmh:         # wenn neuer potentieller Pause-Punkt gefunden wird...
        ptis=idx                                    # ...dann dessen Index in "track" vermerken
        latl=[lat]                                  # ...und Koordinaten-Listen initialisieren
        lonl=[lon]
    return(pause_point)


  # ######################################################################
  # *** test ***
  # Aus der gspxml-Datei kann auch ein Track ausgelesen werden.
  # Dazu sind die Sätze mit BSSID == "GP:SD:TR:AC:KL:OG" heranzuziehen.
  # Allerdings taugen die Daten nicht, um daraus die Geschwindigkeit
  # zwischen vorigem und aktuellen Punkt zu berechnen. Es gibt immer
  # wieder Sprünge (Zeit steigt halbwegs gleichmäßig, Entfernung springt).
  # Vielleicht liegts daran, dass das GPS-Modul von Kismet und dem 
  # Tracker-Script gleichzeitig abgefragt wird.  How auch ever....
  # In den Sätzen gibt es dazu ein "spd"-Tag, das (vermeintlich) die
  # Geschwindigkeit in Meter pro Sekunde angibt.
  def parseGpsXml(self, gpsxml, ignore_MAC_addresses=[]):
    with open(gpsxml, "r") as fl:
      filedata=fl.read()
    trk=list()
    pts=list()
    root=etree.fromstring(filedata.encode())
    for point in root.iter("gps-run"):              # da sollte es nur einen von geben, aber wer weiss
      lvl1=point.getchildren()                      # eine Ebene runter
      for e in lvl1:                                # über alle gps-point's
        if e.tag.lower()=="gps-point":
          bssid=e.get("bssid")
          if bssid=="GP:SD:TR:AC:KL:OG":
            spd=float(e.get("spd"))
            if spd>=0.0:
              t=float(e.get("time-sec"))+float("0."+e.get("time-usec"))
              stime=datetime.fromtimestamp(t)
              lat=float(e.get("lat"))
              lon=float(e.get("lon"))
              trk.append(((lat, lon), (t, spd*3.6, 0.0, 0)))
              #pts.append(((lat, lon), 3, (255, 0, 0), (99, " %s \n %.2f \n %s "%(bssid, spd*3.6, stime))))
          elif bssid not in ignore_MAC_addresses:
            t=int(e.get("time-sec"))
            stime=datetime.fromtimestamp(t)
            lat=float(e.get("lat"))
            lon=float(e.get("lon"))
            dbm=int(e.get("signal_dbm"))
            pts.append(((lat, lon), 3, (255, 0, 0), (99, " %s \n %d \n %s "%(bssid, dbm, stime))))
    return(trk, pts)


  # ######################################################################
  # Analog zu calcGpxTrackpointData() - nur ohne Geschwindigkeit
  def calcGpsxmlTrackpointData(self, trackLL):
    lst=list()
    if len(trackLL)==0:
      return(lst)
    prev_posLL=None
    dist, secs=(0.0, 0)
    for posLL, data in trackLL:
      if prev_posLL is not None:                  # ab dem zweiten Wert...
        c_dist=self.haversine(prev_posLL, posLL)  # Entfernung zwischen n-1 und n
        c_secs=data[0]-pref_timestmp              # Dauer von n-1 bis n
        dist+=c_dist
        secs+=c_secs
      lst.append((posLL, (data[0], data[1], dist, secs)))
      prev_posLL, pref_timestmp=posLL, data[0]
    return(lst)


  # ######################################################################
  # Analog zu calcGpxTrackpointData() - für DB, ohne Geschwindigkeit
  def calcDbTrackpointData(self, trackLL):
    lst=list()
    if len(trackLL)==0:
      return(lst)
    prev_posLL=None
    dist, secs=(0.0, 0)
    for lat, lon, timestamp, speed, altitude, heading in trackLL:
      posLL=(lat, lon)
      timestamp=int(datetime.strptime(timestamp, "%Y-%m-%d %H:%M:%S.%f").strftime("%s"))
      if prev_posLL is not None:                  # ab dem zweiten Wert...
        c_dist=self.haversine(prev_posLL, posLL)  # Entfernung zwischen n-1 und n
        c_secs=timestamp-pref_timestmp            # Dauer von n-1 bis n
        dist+=c_dist
        secs+=c_secs
      lst.append((posLL, (timestamp, speed, dist, secs, altitude, heading)))
      prev_posLL, pref_timestmp=posLL, timestamp
    return(lst)



  # ###########################################################
  # Liefert für eine Lat/Lon-Koordinate die Delta-Werte für
  # die Distanz von einem Kilometer.
  def findDeltaFor1km(self, (lat, lon)):
    dlat, dlon=1.0, 1.0
    dslat, dslon=dlat, dlon
    for x in range(20): # 20 Iterationen langen schon für eine Genauigkeit von 0.1 Meter 
      lat2=lat+dlat
      d=haversine((lat, lon), (lat2, lon))
      dslat/=2
      if d>=1.0:  dlat-=dslat
      else:       dlat+=dslat
      lon2=lon+dlon
      d=haversine((lat, lon), (lat, lon2))
      dslon/=2
      if d>=1.0:  dlon-=dslon
      else:       dlon+=dslon
      #print "%.8f %.8f"%(dlat, dlon)
    return((dlat, dlon))


  # ###########################################################
  # Liefert für eine Lat/Lon-Koordinate die Delta-Werte für
  # die Distanz. Die Distanz muss unterhalb von lat+-2.0 und
  # lon+-2.0 liegen. Ergo: gut bis 100km.
  def findLatLonDeltaForDistance(self, (lat, lon), dist_in_meters):
    dlat, dlon=1.0, 1.0
    dslat, dslon=dlat, dlon
    for x in range(25): # 25 Iterationen langen schon für eine Genauigkeit von 0.1 Meter 
      lat2=lat+dlat
      d=self.haversine((lat, lon), (lat2, lon))*1000
      dslat/=2.0
      if d>=dist_in_meters: dlat-=dslat
      else:                 dlat+=dslat
      lon2=lon+dlon
      d=self.haversine((lat, lon), (lat, lon2))*1000
      dslon/=2.0
      if d>=dist_in_meters: dlon-=dslon
      else:                 dlon+=dslon
      #print "%.8f %.8f"%(dlat, dlon)
    return((dlat, dlon))



# ######################################################################
# Tests
if __name__=="__main__":
  c=OSMhelper()
  t, p=c.parseGpsXml()
  print t
  print
  print p
  sys.exit()
  
  trks=c.loadGPX("/home/dede/wd/xml_tracker/2016/2016_09_25_12_39_25.gpx")

  for trk in trks:
    print c.getPausePoints(trk)

  if False:
    print "-"*60
    pause_point=list()
    lato=lono=tmo=None
    pp_first=pp_last=None
    avg_spd=0.0
    spd_cnt=0
    max_spd=0.0
    for (lat, lon), tm in trk:
      if lato is None:
        dist=0
        dur=0
        lato, lono=lat, lon
        tmo=tm
        spd=0
      else:
        dist=c.haversine((lato, lono), (lat, lon))
        dur=tm-tmo
        spd=dist/(dur/3600.0)
        lato, lono, tmo=lat, lon, tm
      max_spd=max(max_spd, spd)
      if spd>2:
        avg_spd+=spd
        spd_cnt+=1
      if pp_first is None and spd<5:
        pp_first=(lat, lon, tm)
      if pp_first is not None:
        if spd>5:
          if (tm-pp_first[2])>120:
            pause_point.append((pp_first, pp_last))
          pp_first=None
        else:
          pp_last=(lat, lon, tm)
      print "%14.11f %14.11f %11d % 4dm % 3ds %5.1dkm/h"%(lat, lon, tm, dist*1000, dur, spd)
    
    print "max. speed=%4.1fkm/h"%(max_spd)
    print "avg. speed=%4.1fkm/h"%(avg_spd/spd_cnt)
    print pause_point
